﻿using Microsoft.Win32.SafeHandles;
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Text;

namespace CryptoProvider.CryptoAPI
{
    internal class AlgorithmsFinder
    {
        const int PP_ENUMALGS = 1;
        const int CRYPT_FIRST = 1;
        const int CRYPT_OID_INFO_ALGID_KEY = 3;
        const uint CRYPT_VERIFYCONTEXT = 0xF0000000;

        internal static HashSet<string> FindAllSupportedAlgorithms()
        {
            int dwIndex = 0;
            int dwType = 0;
            int pcbTypeName = 0;

            var oids = new HashSet<string>();

            while (CryptEnumProviderTypes(dwIndex, 0, 0, ref dwType, null, ref pcbTypeName))
            {
                if (CryptEnumProviderTypes(dwIndex++, 0, 0, ref dwType, null, ref pcbTypeName))
                {
                    ProvHandle hProv = null;
                    try
                    {
                        if (!CryptAcquireContext(out hProv, null, null, dwType, CRYPT_VERIFYCONTEXT))
                            continue;
                        FindAlgorithms(hProv, oids);
                    }
                    catch (Exception e)
                    {
                    }
                    finally
                    {
                        hProv?.Dispose();
                    }
                }

            }
            return oids;
        }

        private static void FindAlgorithms(ProvHandle hProv, HashSet<string> oids)
        {
            bool fMore = true;
            int cbData = 1024;
            int dwFlags = CRYPT_FIRST;
            while (fMore)
            {
                var pbData = Marshal.AllocHGlobal(cbData);
                try
                {
                    if (CryptGetProvParam(hProv, PP_ENUMALGS, pbData, ref cbData, dwFlags))
                    {
                        dwFlags = 0;
                        var oid = GetOid(pbData);
                        if (!string.IsNullOrEmpty(oid))
                            oids.Add(oid);
                    }
                    else
                    {
                        fMore = false;
                    }
                }
                catch (Exception e)
                {
                }
                finally
                {
                    Marshal.FreeHGlobal(pbData);
                }
            }
        }

        private static string GetOid(IntPtr alg_id)
        {
            IntPtr info = CryptFindOIDInfo(CRYPT_OID_INFO_ALGID_KEY, alg_id, 0);
            if (info == IntPtr.Zero)
                return null;

            CRYPT_OID_INFO oidInfo = Marshal.PtrToStructure<CRYPT_OID_INFO>(info);
            var oid = Marshal.PtrToStringAnsi(oidInfo.pszOID);
            return oid;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct CRYPT_OID_INFO
        {
            public int cbSize;
            public IntPtr pszOID;
            public IntPtr pwszName;
            public int dwGroupId;
            public int dwValue;
            public IntPtr pwszExtension;
        }

        private sealed class ProvHandle : SafeHandleZeroOrMinusOneIsInvalid
        {
            public ProvHandle()
                : base(true)
            {
            }

            protected override bool ReleaseHandle()
            {
                return CryptReleaseContext(handle, 0);
            }

            [DllImport("advapi32.dll", SetLastError = true)]
            private static extern bool CryptReleaseContext(IntPtr hProv, int dwFlags);

        }

        [DllImport("advapi32.dll", SetLastError = true)]
        static extern bool CryptEnumProviderTypes(int dwIndex, int pdwReserved, int dwFlags, [In] ref int pdwProvType, StringBuilder pszTypeName, [In] ref int pcbTypeName);

        [DllImport("advapi32.dll", SetLastError = true)]
        private static extern bool CryptAcquireContext(out ProvHandle phProv, string pszContainer, string pszProvider, int dwProvType, uint dwFlags);

        [DllImport("advapi32.dll", SetLastError = true)]
        private static extern bool CryptGetProvParam(ProvHandle hProv, int dwParam, IntPtr pbData, ref int pdwDataLen, int dwFlags);

        [DllImport("Crypt32.dll", SetLastError = true)]
        public static extern IntPtr CryptFindOIDInfo(uint dwKeyType, IntPtr pvKey, uint dwGroupId);
    }
}
