﻿using Ascon.Pilot.SDK;
using Org.BouncyCastle.Asn1;
using Org.BouncyCastle.Asn1.Cms;
using Org.BouncyCastle.Asn1.Pkcs;
using Org.BouncyCastle.Cms;
using Org.BouncyCastle.Tsp;
using System.Linq;
using System.Security.Cryptography;
using System.Security.Cryptography.Pkcs;

namespace CryptoProvider.CryptoAPI
{
    internal class Asn1Helper
    {
        public static System.DateTime? GetSignDate(System.Security.Cryptography.Pkcs.SignerInfo signerInfo)
        {
            if (signerInfo.SignedAttributes == null)
                return null;

            var signDateOid = PkcsObjectIdentifiers.Pkcs9AtSigningTime.ToString();
            foreach (var attr in signerInfo.SignedAttributes)
            {
                if (attr.Oid.Value == signDateOid) 
                {
                    AsnEncodedData asnData = attr.Values[0];
                    Pkcs9SigningTime pkcs9SigningTime = new Pkcs9SigningTime(asnData.RawData);
                    return pkcs9SigningTime.SigningTime;
                }
            }
            return null;
        }

        public static TimeStampToken GetTimestampToken(System.Security.Cryptography.Pkcs.SignerInfo signerInfo)
        {
            foreach (var attr in signerInfo.UnsignedAttributes)
            {
                if (attr.Oid.Value == PkcsObjectIdentifiers.IdAASignatureTimeStampToken.Id)
                {
                    var timeStampAttribute = attr.Values[0];
                    var cms = new CmsSignedData(timeStampAttribute.RawData);
                    return new TimeStampToken(cms);
                }
            }
            return null;
        }

        public static byte[] GetTimeStampCertificate(TimeStampToken token)
        {
            var asnCertificateSet = token.UnsignedAttributes.GetAll(PkcsObjectIdentifiers.IdAAEtsCertValues);
            var attributeCerts = (Attribute)asnCertificateSet.First();
            var asnObjCerts = (Asn1Sequence)attributeCerts.AttrValues[0];

            var firstCertOfChanin = asnObjCerts[0];
            return firstCertOfChanin.GetEncoded();
        }

        public static CadesType GetCadesType(byte[] signature, bool isSelfSigned)
        {
            var signedData = new CmsSignedData(signature);
            var signer = signedData.GetSignerInfos().GetSigners().First();
            return GetCadesType(signer, isSelfSigned);
        }

        private static CadesType GetCadesType(SignerInformation signerInfo, bool isSelfSignedCertificate)
        {
            // CADES-BES — базовая подпись, без штампов времени и ссылок на сертификаты.
            // CADES-EPES — добавляется политика подписи.
            // CADES-T — добавляется штамп времени.
            // CADES-C — добавляются полные значения сертификатов и CRL для самодостаточной проверки.
            // CADES-X Long Type 1 — добавляет ссылки на сертификаты и CRL, а также архивный штамп времени.
            // CADES-X Long Type 2 — добавляет полные значения сертификатов и CRL(как в CADES-C).
            // CADES-XL
            // CADES-A — архивная подпись, включающая все необходимые данные для долговременной проверки подписи.

            // is CADES-BES
            var type = CadesType.CadesBes;
            var isCadesBes = true;

            if (signerInfo.UnsignedAttributes == null)
                return type;

            // CADES-EPES
            var etsPolicy = signerInfo.UnsignedAttributes.GetAll(PkcsObjectIdentifiers.IdAAEtsSigPolicyID)?.Count > 0;
            var isCadesEpes = isCadesBes && etsPolicy;

            type = isCadesEpes ? CadesType.CadesEpes : type;

            // CADES-T
            var timeStampToken = signerInfo.UnsignedAttributes.GetAll(PkcsObjectIdentifiers.IdAASignatureTimeStampToken);
            var isCadesT = isCadesBes && timeStampToken != null && timeStampToken.Count > 0;
            type = isCadesT ? CadesType.CadesT : type;

            // CADES-C
            // IdAAEtsRevocationValues - опционально
            // Только ссылки на сертификаты и список отозванных
            var certRefs = signerInfo.UnsignedAttributes.GetAll(PkcsObjectIdentifiers.IdAAEtsCertificateRefs);
            var revocationRefs = signerInfo.UnsignedAttributes.GetAll(PkcsObjectIdentifiers.IdAAEtsRevocationRefs);
            var isCadesC = isCadesT && revocationRefs?.Count > 0 && certRefs?.Count > 0;
            type = isCadesC ? CadesType.CadesC : type;


            //CADES-X Long
            //получаем из CADES-C + атрибуты certificate-values и revocation-values
            var certificateValues = signerInfo.UnsignedAttributes.GetAll(PkcsObjectIdentifiers.IdAAEtsCertValues)?.Count > 0;
            var revocationValues = signerInfo.UnsignedAttributes.GetAll(PkcsObjectIdentifiers.IdAAEtsRevocationValues)?.Count > 0;

            var isCadesXLong = isCadesC && revocationValues && certificateValues;
            if (!isCadesXLong)
                return type;

            // CADES-X Long Type 1
            // получаем из CADES-X Long + атрибут CAdES-C-time-stamp, который содержит штамп времени на всей подписи CAdES-C
            var escTimeStamp = signerInfo.UnsignedAttributes.GetAll(PkcsObjectIdentifiers.IdAAEtsEscTimeStamp)?.Count > 0;
            var isCadesXlt1 = isCadesXLong && escTimeStamp;
            type = isCadesXlt1 ? CadesType.CadesXLT1 : type;

            // CADES-X Long Type 2
            // получаем из CADES-X Long + штамп времени на ссылки на сертификаты и списки отзыва сертификатов. 
            var escCertTimeStamp = signerInfo.UnsignedAttributes.GetAll(PkcsObjectIdentifiers.IdAAEtsCertCrlTimestamp)?.Count > 0;
            var isCadesXlt2 = isCadesXLong && escCertTimeStamp;
            type = isCadesXlt2 ? CadesType.CadesXLT2 : type;

            // CADES-A
            // получаем из CADES-X Long Type 1 или CADES-X Long Type 2 + archiveTimeStamp
            var archiveTimeStamp = signerInfo.UnsignedAttributes.GetAll(PkcsObjectIdentifiers.IdAAEtsArchiveTimestamp)?.Count > 0;
            var isCadesA = (isCadesXlt1 || isCadesXlt2) && archiveTimeStamp;
            type = isCadesA ? CadesType.CadesA : type;

            return type;
        }
    }
}
