/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Input;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Controls;
using Ascon.Pilot.SDK.Controls.Commands;
using Ascon.Pilot.SDK.ObjectCard;

namespace CustomXpsAnnotations
{
    class CustomXpsAnnotationViewModel : PropertyChangedBase
    {
        private readonly XpsAnnotation _annotation;
        private readonly IRemarkModifier _modifier;
        private readonly IObjectsRepository _repository;
        private readonly IObjectCardControlProvider _objectCardControlProvider;
        private readonly DelegateCommand _deleteCommand;
        private readonly DelegateCommand _saveCommand;
        private readonly DelegateCommand _cancelCommand;

        private double _x;
        private double _y;
        private string _mark;
        private string _content;
        private bool _isSelected;
        private bool _isPopupOpen;
        private ObjectCardControl _objectCard;

        public CustomXpsAnnotationViewModel(XpsAnnotation annotation, IRemarkModifier modifier,
            IObjectsRepository repository, IObjectCardControlProvider objectCardControlProvider)
        {
            _annotation = annotation;
            _modifier = modifier;
            _repository = repository;
            _objectCardControlProvider = objectCardControlProvider;

            X = annotation.PositionX;
            Y = annotation.PositionY;
            Mark = annotation.Mark;
            
            _deleteCommand = new DelegateCommand(Delete, CanDelete)
            {
                Caption = "Delete"
            };
            _cancelCommand = new DelegateCommand(Cancel);
            _saveCommand = new DelegateCommand(Save, CanSave);

            LoadObjectCard(annotation.AnnotationId);
        }

        public ICommand DeleteCommand => _deleteCommand;
        public ICommand SaveCommand => _saveCommand;
        public ICommand CancelCommand => _cancelCommand;

        public string Author { get; }

        public double X
        {
            get => _x;
            set
            {
                _x = value;
                NotifyOfPropertyChange(nameof(X));
            }
        }

        public double Y
        {
            get => _y;
            set
            {
                _y = value;
                NotifyOfPropertyChange(nameof(Y));
            }
        }

        public string Mark
        {
            get => _mark;
            set
            {
                _mark = value;
                NotifyOfPropertyChange(nameof(Mark));
            }
        }        
        
        public string Content
        {
            get => _content;
            set
            {
                _content = value;
                NotifyOfPropertyChange(nameof(Content));
            }
        }

        public bool IsSelected
        {
            get => _isSelected; 
            set
            {
                _isSelected = value;
                NotifyOfPropertyChange(nameof(IsSelected));

                if (!_isSelected)
                    IsPopupOpen = false;
            }
        }

        public bool IsPopupOpen
        {
            get => _isPopupOpen;
            set
            {
                _isPopupOpen = value;
                NotifyOfPropertyChange(nameof(IsPopupOpen));
            }
        }

        public ObjectCardControl ObjectCard
        {
            get => _objectCard;
            set
            {
                _objectCard = value;
                NotifyOfPropertyChange(nameof(ObjectCard));
            }
        }

        private bool CanDelete()
        {
            var person = _repository.GetCurrentPerson();
            var accessLevel = _repository.GetCurrentAccess(_annotation.AnnotationId, person.Id);
            return accessLevel.HasFlag(AccessLevel.Edit);
        }

        private void Delete()
        {
            _modifier.DeleteRemark(_annotation.AnnotationId);
            _modifier.Apply();
        }

        public void ShowPopup()
        {
            IsPopupOpen = true;
        }

        private void Save()
        {
            var builder = _modifier.EditRemark(_annotation.AnnotationId);
            foreach (var value in ObjectCard.Interaction.GetValues())
            {
                builder.SetAttributeAsObject(value.Key, value.Value.Value);
            }
            _modifier.Apply();

            LoadObjectCard(_annotation.AnnotationId);
        }

        private bool CanSave()
        {
            var person = _repository.GetCurrentPerson();
            var accessLevel = _repository.GetCurrentAccess(_annotation.AnnotationId, person.Id);
            return ObjectCard != null && ObjectCard.Interaction.GetIsDataChanged() && accessLevel.HasFlag(AccessLevel.Edit);
        }

        private void Cancel()
        {
            IsSelected = false;
        }

        private void LoadObjectCard(Guid annotationId)
        {
            if (ObjectCard != null)
                ObjectCard.Interaction.OnObjectCardUpdated -= OnValidInputChanged;

            ObjectCard = _objectCardControlProvider.GetObjectCardControl(annotationId, !CanDelete());
            ObjectCard.Interaction.OnObjectCardUpdated += OnValidInputChanged;
            _saveCommand.RaiseCanExecuteChanged();
            Content = string.Join(" - ", ObjectCard.Interaction.GetValues().Select(x => x.Value.Value));
        }

        private void OnValidInputChanged(object sender, EventArgs e)
        {
            _saveCommand.RaiseCanExecuteChanged();
        }
    }
}
