/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Threading;
using System.Threading.Tasks;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Controls.ObjectCardView;
using Ascon.Pilot.SDK.Extensions;
using DynamicData;
using Moq;
using NUnit.Framework;
using ProjectCloneWizard.Common;
using ProjectCloneWizard.Data;
using ProjectCloneWizard.ObjectsTree;
using ProjectCloneWizard.ViewModels;

namespace ProjectCloneWizard.Tests
{
    [TestFixture]
    class ProcessPageViewModelTests
    {
        private Mock<IDataObjectService> _dataObjectServiceMock;
        private Mock<IObjectModifier> _objectModifierMock;
        private Mock<IObjectsRepository> _objectRepositoryMock;
        private Mock<IFileProvider> _fileProviderMock;
        private Mock<IObjectBuilder> _objectBuilder;

        [SetUp]
        public void SetUp()
        {
            _dataObjectServiceMock = new Mock<IDataObjectService>();
            var dataObjects = new SourceCache<IDataObject, Guid>(x => x.Id);
            _dataObjectServiceMock.Setup(s => s.DataObjects).Returns(dataObjects.AsObservableCache);

            _objectModifierMock = new Mock<IObjectModifier>();
            _objectRepositoryMock = new Mock<IObjectsRepository>();
            _objectRepositoryMock.Setup(r => r.GetCurrentPerson()).Returns(TestTools.GetIPersonMock().Object);
            _fileProviderMock = new Mock<IFileProvider>();
            _objectBuilder = new Mock<IObjectBuilder>();
        }

        [Test]
        public async Task should_call_apply_once()
        {
            //given
            var creationInfo = NewCreationInfo();

            //when
            var dataObject = TestTools.GetIDataObjectMock().Object;
            var list = new List<IDataObject> { dataObject };
            var res = new Task<IList<IDataObject>>(() => list);
            await StartViewModel(creationInfo, res);

            //then
            _objectModifierMock.Verify(m => m.Apply(), Times.Once);
        }

        [Test]
        public async Task should_add_access_to_objects()
        {
            //given
            var creationInfo = NewCreationInfo();
            var dataObjectMock = TestTools.GetIDataObjectMock();
            var accessRecord = new Mock<IAccessRecord>();
            accessRecord.Setup(a => a.Access).Returns(new Mock<IAccess>().Object);
            var records = new List<IAccessRecord> { accessRecord.Object };

            dataObjectMock.Setup(o => o.Id).Returns(Guid.NewGuid());
            dataObjectMock.Setup(o => o.ParentId).Returns(creationInfo.Template.Id);
            dataObjectMock.Setup(o => o.Access2).Returns(new ReadOnlyCollection<IAccessRecord>(records));

            var nodeViewMode = TestTools.GetNodeViewModel(dataObjectMock, _dataObjectServiceMock);
            var collection = new ObservableCollection<NodeViewModel> {nodeViewMode};
            creationInfo.ObjectsNodes = new ReadOnlyObservableCollection<NodeViewModel>(collection);

            creationInfo.CopyAccessForObjects = true;

            //when
            var res = new Task<IList<IDataObject>>(() => new List<IDataObject> { dataObjectMock.Object });
            await StartViewModel(creationInfo, res);

            //then
            _objectBuilder.Verify(m => m.AddAccessRecords(It.IsAny<int>(), It.IsAny<AccessLevel>(), It.IsAny<DateTime>(), AccessInheritance.None, AccessType.Allow, It.IsAny<int[]>()), Times.AtLeastOnce);
        }

        [Test]
        public async Task should_add_access_to_objects_deny()
        {
            //given
            var creationInfo = NewCreationInfo();
            var dataObjectMock = TestTools.GetIDataObjectMock();
            var access = new Mock<IAccess>();
            access.Setup(x => x.Type).Returns(AccessType.Deny);
            access.Setup(x => x.TypeIds).Returns(new[] { 1, 2, 3 });
            var accessRecord = new Mock<IAccessRecord>();
            accessRecord.Setup(a => a.Access).Returns(access.Object);
            var records = new List<IAccessRecord> { accessRecord.Object };

            dataObjectMock.Setup(o => o.Id).Returns(Guid.NewGuid());
            dataObjectMock.Setup(o => o.ParentId).Returns(creationInfo.Template.Id);
            dataObjectMock.Setup(o => o.Access2).Returns(new ReadOnlyCollection<IAccessRecord>(records));

            var nodeViewMode = TestTools.GetNodeViewModel(dataObjectMock, _dataObjectServiceMock);
            var collection = new ObservableCollection<NodeViewModel> {nodeViewMode};
            creationInfo.ObjectsNodes = new ReadOnlyObservableCollection<NodeViewModel>(collection);

            creationInfo.CopyAccessForObjects = true;

            //when
            var res = new Task<IList<IDataObject>>(() => new List<IDataObject> { dataObjectMock.Object });
            await StartViewModel(creationInfo, res);

            //then
            _objectBuilder.Verify(m => m.AddAccessRecords(It.IsAny<int>(), It.IsAny<AccessLevel>(), It.IsAny<DateTime>(), AccessInheritance.None, AccessType.Deny, new[] { 1, 2, 3 }), Times.AtLeastOnce);
        }

        [Test]
        public async Task should_no_access_to_objects()
        {
            //given
            var creationInfo = NewCreationInfo();
            var dataObjectMock = TestTools.GetIDataObjectMock();
            
            var access = new List<IAccessRecord> {new Mock<IAccessRecord>().Object};

            dataObjectMock.Setup(o => o.Id).Returns(Guid.NewGuid());
            dataObjectMock.Setup(o => o.ParentId).Returns(creationInfo.Template.Id);
            dataObjectMock.Setup(o => o.Access2).Returns(access.AsReadOnly);

            var nodeViewMode = TestTools.GetNodeViewModel(dataObjectMock, _dataObjectServiceMock);
            var collection = new ObservableCollection<NodeViewModel> { nodeViewMode };
            creationInfo.ObjectsNodes = new ReadOnlyObservableCollection<NodeViewModel>(collection);

            //when
            creationInfo.CopyAccessForObjects = false;
            var res = new Task<IList<IDataObject>>(() => new List<IDataObject> { dataObjectMock.Object });
            await StartViewModel(creationInfo, res);

            //then
            _objectBuilder.Verify(m => m.AddAccessRecords(It.IsAny<int>(), It.IsAny<AccessLevel>(), It.IsAny<DateTime>(), AccessInheritance.None, AccessType.Allow), Times.Never);
        }

        [Test]
        public async Task should_add_access_to_files()
        {
            //given
            var creationInfo = NewCreationInfo();
            var dataObjectMock = TestTools.GetIDataObjectMock();
            var accessRecord = new Mock<IAccessRecord>();
            accessRecord.Setup(a => a.Access).Returns(new Mock<IAccess>().Object);
            var records = new List<IAccessRecord> { accessRecord.Object };

            dataObjectMock.Setup(o => o.Id).Returns(Guid.NewGuid());
            dataObjectMock.Setup(o => o.ParentId).Returns(creationInfo.Template.Id);
            dataObjectMock.Setup(o => o.Access2).Returns(new ReadOnlyCollection<IAccessRecord>(records));

            var nodeViewMode = TestTools.GetNodeViewModel(dataObjectMock, _dataObjectServiceMock);
            var collection = new ObservableCollection<NodeViewModel> { nodeViewMode };
            creationInfo.StorageNodes = new ReadOnlyObservableCollection<NodeViewModel>(collection);

            //when
            creationInfo.CopyAccessForFiles = true;
            creationInfo.CopyAccessForObjects = false;
            var res = new Task<IList<IDataObject>>(() => new List<IDataObject> { dataObjectMock.Object });
            await StartViewModel(creationInfo, res);

            //then
            _objectBuilder.Verify(m => m.AddAccessRecords(It.IsAny<int>(), It.IsAny<AccessLevel>(), It.IsAny<DateTime>(), AccessInheritance.None, AccessType.Allow, It.IsAny<int[]>()), Times.Once);
        }

        [Test]
        public async Task should_add_access_to_files_and_objects()
        {
            //given
            var creationInfo = NewCreationInfo();
            var dataObjectMock = TestTools.GetIDataObjectMock();
            var accessRecord = new Mock<IAccessRecord>();
            accessRecord.Setup(a => a.Access).Returns(new Mock<IAccess>().Object);
            var records = new List<IAccessRecord> { accessRecord.Object };

            dataObjectMock.Setup(o => o.Id).Returns(Guid.NewGuid());
            dataObjectMock.Setup(o => o.ParentId).Returns(creationInfo.Template.Id);
            dataObjectMock.Setup(o => o.Access2).Returns(new ReadOnlyCollection<IAccessRecord>(records));

            var nodeViewMode = TestTools.GetNodeViewModel(dataObjectMock, _dataObjectServiceMock);
            var collection = new ObservableCollection<NodeViewModel> { nodeViewMode };
            creationInfo.StorageNodes = new ReadOnlyObservableCollection<NodeViewModel>(collection);
            creationInfo.ObjectsNodes = new ReadOnlyObservableCollection<NodeViewModel>(collection);

            //when
            creationInfo.CopyAccessForFiles = true;
            creationInfo.CopyAccessForObjects = true;
            var res = new Task<IList<IDataObject>>(() => new List<IDataObject> { dataObjectMock.Object });
            await StartViewModel(creationInfo, res);

            //then
            _objectBuilder.Verify(m => m.AddAccessRecords(It.IsAny<int>(), It.IsAny<AccessLevel>(), It.IsAny<DateTime>(), AccessInheritance.None, AccessType.Allow, It.IsAny<int[]>()), Times.Exactly(2));
        }

        [Test]
        public async Task should_add_only_checked_files()
        {
            //given
            var creationInfo = NewCreationInfo();

            // 1 объект в дереве документов
            var id = Guid.NewGuid();
            var type = TestTools.GetITypeMock();
            type.Setup(t => t.Name).Returns("Type1");
            var dataObjectMock = TestTools.GetIDataObjectMock(type, id);
            var objectNodeViewMode = TestTools.GetNodeViewModel(dataObjectMock, _dataObjectServiceMock);
            var collection = new ObservableCollection<NodeViewModel> { objectNodeViewMode };
            creationInfo.ObjectsNodes = new ReadOnlyObservableCollection<NodeViewModel>(collection);

            // 1 объект в storage
            var strorageId = Guid.NewGuid();
            var storageType = TestTools.GetITypeMock();
            storageType.Setup(t => t.Name).Returns("StorageType");
            var dataStorageObjectMock = TestTools.GetIDataObjectMock(storageType, strorageId);
            var storageNodeViewMode = TestTools.GetNodeViewModel(dataStorageObjectMock, _dataObjectServiceMock);
            var storageCollection = new ObservableCollection<NodeViewModel> { storageNodeViewMode };
            creationInfo.StorageNodes = new ReadOnlyObservableCollection<NodeViewModel>(storageCollection);


            _objectRepositoryMock.Setup(r => r.GetTypes()).Returns(new[] { type.Object, storageType.Object });
            //when
            objectNodeViewMode.IsChecked = false;
            storageNodeViewMode.IsChecked = true;

            var res = new Task<IList<IDataObject>>(() => new List<IDataObject> { dataObjectMock.Object, dataStorageObjectMock.Object });
            await StartViewModel(creationInfo, res);

            //then
            //2 раза должен вызваться метод создания объекта.
            //1 - проект
            _objectModifierMock.Verify(m => m.CreateById(It.IsAny<Guid>(), It.IsAny<Guid>(), It.IsAny<IType>()), Times.Exactly(2));
            //1- объект
            _objectModifierMock.Verify(m => m.CreateById(It.IsAny<Guid>(), It.IsAny<Guid>(), storageType.Object), Times.Once);
            _objectModifierMock.Verify(m => m.CreateById(It.IsAny<Guid>(), It.IsAny<Guid>(), type.Object), Times.Never);
        }

        [Test]
        public async Task should_add_only_checked_objects()
        {
            //given
            var creationInfo = NewCreationInfo();

            // 1 объект в дереве документов
            var id = Guid.NewGuid();
            var type = TestTools.GetITypeMock();
            type.Setup(t => t.Name).Returns("Type1");
           
            var dataObjectMock = TestTools.GetIDataObjectMock(type, id);
            var objectNodeViewMode = TestTools.GetNodeViewModel(dataObjectMock, _dataObjectServiceMock);
            var collection = new ObservableCollection<NodeViewModel> { objectNodeViewMode };
            creationInfo.ObjectsNodes = new ReadOnlyObservableCollection<NodeViewModel>(collection);

            // 1 объект в storage
            var strorageId = Guid.NewGuid();
            var storageType = TestTools.GetITypeMock();
            storageType.Setup(t => t.Name).Returns("StorageType");

            var dataStorageObjectMock = TestTools.GetIDataObjectMock(storageType, strorageId);
            var storageNodeViewMode = TestTools.GetNodeViewModel(dataStorageObjectMock, _dataObjectServiceMock);
            var storageCollection = new ObservableCollection<NodeViewModel> { storageNodeViewMode };
            creationInfo.StorageNodes = new ReadOnlyObservableCollection<NodeViewModel>(storageCollection);

            _objectRepositoryMock.Setup(r => r.GetTypes()).Returns(new[] { type.Object, storageType.Object });
            //when
            objectNodeViewMode.IsChecked = true;
            storageNodeViewMode.IsChecked = false;

            var res = new Task<IList<IDataObject>>(() => new List<IDataObject> { dataObjectMock.Object, dataStorageObjectMock.Object });
            await StartViewModel(creationInfo, res);

            //then
            //2 раза должен вызваться метод создания объекта.
            //1 - проект
            _objectModifierMock.Verify(m => m.CreateById(It.IsAny<Guid>(), It.IsAny<Guid>(), It.IsAny<IType>()), Times.Exactly(2));
            //1- объект
            _objectModifierMock.Verify(m => m.CreateById(It.IsAny<Guid>(), It.IsAny<Guid>(), type.Object), Times.Once);
            _objectModifierMock.Verify(m => m.CreateById(It.IsAny<Guid>(), It.IsAny<Guid>(), storageType.Object), Times.Never);
        }

        [Test]
        public async Task should_not_add_any_objects()
        {
            //given
            var creationInfo = NewCreationInfo();

            // 1 объект в дереве документов
            var id = Guid.NewGuid();
            var type = TestTools.GetITypeMock();
            type.Setup(t => t.Name).Returns("Type1");
            _objectRepositoryMock.Setup(r => r.GetType("Type1")).Returns(type.Object);

            var dataObjectMock = TestTools.GetIDataObjectMock(type, id);
            var objectNodeViewMode = TestTools.GetNodeViewModel(dataObjectMock, _dataObjectServiceMock);
            var collection = new ObservableCollection<NodeViewModel> { objectNodeViewMode };
            creationInfo.ObjectsNodes = new ReadOnlyObservableCollection<NodeViewModel>(collection);

            // 1 объект в storage
            var strorageId = Guid.NewGuid();
            var storageType = TestTools.GetITypeMock();
            storageType.Setup(t => t.Name).Returns("StorageType");
            _objectRepositoryMock.Setup(r => r.GetType("StorageType")).Returns(storageType.Object);
            var dataStorageObjectMock = TestTools.GetIDataObjectMock(storageType, strorageId);
            var storageNodeViewMode = TestTools.GetNodeViewModel(dataStorageObjectMock, _dataObjectServiceMock);
            var storageCollection = new ObservableCollection<NodeViewModel> { storageNodeViewMode };
            creationInfo.StorageNodes = new ReadOnlyObservableCollection<NodeViewModel>(storageCollection);

            //when
            objectNodeViewMode.IsChecked = false;
            storageNodeViewMode.IsChecked = false;

            var res = new Task<IList<IDataObject>>(() => new List<IDataObject> { dataObjectMock.Object, dataStorageObjectMock.Object });
            await StartViewModel(creationInfo, res);

            //then
            //2 раза должен вызваться метод создания объекта.
            //1 - проект
            _objectModifierMock.Verify(m => m.CreateById(It.IsAny<Guid>(), It.IsAny<Guid>(), It.IsAny<IType>()), Times.Once);
            //объекты storage и дерева не создаются
            _objectModifierMock.Verify(m => m.CreateById(It.IsAny<Guid>(), It.IsAny<Guid>(), type.Object), Times.Never);
            _objectModifierMock.Verify(m => m.CreateById(It.IsAny<Guid>(), It.IsAny<Guid>(), storageType.Object), Times.Never);
        }

        [Test]
        public async Task should_create_secret_project()
        {
            //given
            var id = Guid.NewGuid();
            var type = TestTools.GetITypeMock();
            type.Setup(t => t.Name).Returns("Type1");
            _objectRepositoryMock.Setup(r => r.GetType("Type1")).Returns(type.Object);

            var dataObjectMock = TestTools.GetIDataObjectMock(type, id);
            dataObjectMock.Setup(o => o.IsSecret).Returns(true);
            var objectNodeViewMode = TestTools.GetNodeViewModel(dataObjectMock, _dataObjectServiceMock);
            var collection = new ObservableCollection<NodeViewModel> { objectNodeViewMode };

            var creationInfo = NewCreationInfo();
            creationInfo.ObjectsNodes = new ReadOnlyObservableCollection<NodeViewModel>(collection);

            var res = new Task<IList<IDataObject>>(() => new List<IDataObject> { dataObjectMock.Object });
            await StartViewModel(creationInfo, res);

            //then
            //проект секретный
            _objectBuilder.Verify(b => b.MakeSecret(), Times.Once);
        }

        [Test]
        public async Task should_create_not_secret_project()
        {
            //given
            var templateObject = TestTools.GetIDataObjectMock();
            templateObject.Setup(o => o.IsSecret).Returns(false);
            var templateNode = TestTools.GetNodeViewModel(templateObject, _dataObjectServiceMock).Source;

            var creationInfo = NewCreationInfo();
            creationInfo.Template = templateNode;

            var res = new Task<IList<IDataObject>>(() => new List<IDataObject>());
            await StartViewModel(creationInfo, res);

            //then
            //проект секретный
            _objectBuilder.Verify(b => b.MakeSecret(), Times.Never);
        }

        [Test]
        public async Task should_create_secret_objects()
        {
            //given
            var templateObject = TestTools.GetIDataObjectMock();
            templateObject.Setup(o => o.IsSecret).Returns(true);
            var templateNode = TestTools.GetNodeViewModel(templateObject, _dataObjectServiceMock);

            var creationInfo = NewCreationInfo();
            creationInfo.Template = templateNode.Source;

            var res = new Task<IList<IDataObject>>(() => new List<IDataObject>());
            await StartViewModel(creationInfo, res);

            //then
            //проект секретный
            _objectBuilder.Verify(b => b.MakeSecret(), Times.Once);
        }

        private static CreationInfo NewCreationInfo()
        {
            var creationInfo = new CreationInfo
            {
                Template = TestTools.GetNodeViewModel().Source,
                Parent = TestTools.GetIDataObjectMock().Object,
                Attributes = new Dictionary<string, DValue>(),
                ObjectsNodes = new ReadOnlyObservableCollection<NodeViewModel>(new ObservableCollection<NodeViewModel>())
            };
            return creationInfo;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="creationInfo"></param>
        /// <param name="res">Результаты запроса к Pilot. Moq.</param>
        private async Task StartViewModel(CreationInfo creationInfo, Task<IList<IDataObject>> res)
        {
            //настройка moq объектов
            _objectRepositoryMock.Setup(r => r.GetType(creationInfo.Template.Type.Name)).Returns(TestTools.GetITypeMock().Object);
            _objectModifierMock.Setup(m => m.CreateById(It.IsAny<Guid>(), It.IsAny<Guid>(), It.IsAny<IType>())).Returns(_objectBuilder.Object);
            _objectBuilder.Setup(b => b.SetCreator(It.IsAny<int>())).Returns(_objectBuilder.Object);
            _objectBuilder.Setup(b => b.DataObject).Returns(TestTools.GetIDataObjectMock().Object);

            //moq для метода расширения
            var asyncMethodsMock = new Mock<IAsyncMethods>();
            asyncMethodsMock.Setup(f => f.GetObjectsAsync(It.IsAny<IObjectsRepository>(), It.IsAny<IEnumerable<Guid>>(), It.IsAny<CancellationToken>())).Returns(
                () =>
                {
                    //возвращаем результат запроса к Pilot
                    res.Start();
                    return res;
                });
            AsyncMethodsExtensions.AsyncMethods = asyncMethodsMock.Object;

            var viewModel = new ProcessPageViewModel(creationInfo, _objectModifierMock.Object, _objectRepositoryMock.Object, _fileProviderMock.Object);
            await viewModel.Start();
        }
    }
}
