/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using Ascon.Pilot.SDK;

namespace ProjectCloneWizard.ObjectsTree
{
    //Фильтр только для детей "Обозревателя проектов"
    class ListViewChildrenFilter : IChildrenFilter
    {
        private readonly IObjectsRepository _repository;
        private readonly int _shortcutTypeId;
        public ListViewChildrenFilter(IObjectsRepository repository)
        {
            _repository = repository;
            _shortcutTypeId = GetShortcutTypeId();
        }
        public IEnumerable<Guid> ChildrenFilter(IDataObject dataObject)
        {
            return ChildrenFilters.GetChildrenForListView(dataObject, _repository);
        }

        public bool Filter(IDataObject obj)
        {
            return obj.State == DataState.Loaded && obj.Type.Id != _shortcutTypeId;
        }

        private int GetShortcutTypeId()
        {
            var type = _repository.GetType(SystemTypeNames.SHORTCUT);
            return type?.Id ?? -1;
        }
    }

    //Фильтр только для папок "Обозревателя проектов". Объекты, которые не имеют детей фильтруются
    class ListViewFolderChildrenFilter : IChildrenFilter
    {
        private readonly IObjectsRepository _repository;
        private readonly int _shortcutTypeId;
        public ListViewFolderChildrenFilter(IObjectsRepository repository)
        {
            _repository = repository;
            _shortcutTypeId = GetShortcutTypeId();
        }
        public IEnumerable<Guid> ChildrenFilter(IDataObject dataObject)
        {
            return ChildrenFilters.GetChildrenForListView(dataObject, _repository);
        }

        public bool Filter(IDataObject obj)
        {
            return obj.State == DataState.Loaded &&
                                obj.Type.Children.Count != 0 &&
                                obj.Type.Id != _shortcutTypeId;
        }

        private int GetShortcutTypeId()
        {
            var type = _repository.GetType(SystemTypeNames.SHORTCUT);
            return type?.Id ?? -1;
        }
    }

    //Фильтр только для детей Pilot-Storage
    class StorageChildrenFilter : IChildrenFilter
    {
        private readonly IObjectsRepository _repository;
        
        public StorageChildrenFilter(IObjectsRepository repository)
        {
            _repository = repository;
        }

        public IEnumerable<Guid> ChildrenFilter(IDataObject dataObject)
        {
            return ChildrenFilters.GetChildrenForPilotStorage(dataObject, _repository);
        }

        public bool Filter(IDataObject obj)
        {
            return obj.State == DataState.Loaded;
        }
    }

    //Ничего не фильтруется
    class AllChildrenFilter : IChildrenFilter
    {
        public IEnumerable<Guid> ChildrenFilter(IDataObject dataObject)
        {
            return dataObject.Children;
        }

        public bool Filter(IDataObject obj)
        {
            return obj.State == DataState.Loaded;
        }
    }
}
