/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Threading;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Extensions;
using DynamicData;
using ProjectCloneWizard.Data;

namespace ProjectCloneWizard.ObjectsTree
{
    public class DataObjectService : IDataObjectService
    {
        private readonly IObjectsRepository _repository;
        private readonly IChildrenFilter _childrenFilter;
        private readonly CompositeDisposable _cleanUp = new CompositeDisposable();
        private readonly SourceCache<IDataObject, Guid> _dataObjects = new SourceCache<IDataObject, Guid>(x => x.Id);

        public IObservableCache<IDataObject, Guid> DataObjects => _dataObjects.AsObservableCache();

        public DataObjectService(IObjectsRepository repository, IChildrenFilter childrenFilter)
        {
            _repository = repository;
            _childrenFilter = childrenFilter;
            _cleanUp.Add(_dataObjects);
        }

        public void GetElementStructureAsync(Guid id, Action nothingLoaded, bool isAddParent)
        {
            var disposable = _repository.SubscribeObjects(new[] {id})
                .Where(x => _repository.GetCurrentAccess(x.Id, _repository.GetCurrentPerson().Id).HasFlag(AccessLevel.View))
                .Where(x => _childrenFilter.Filter(x))
                .SubscribeOnDispatcher(DispatcherPriority.Background)
                .Subscribe(o =>
                {
                    var children = _childrenFilter.ChildrenFilter(o).ToList();
                    if (children.Count == 0)
                    {
                        nothingLoaded();
                        return;
                    }

                    if (isAddParent)
                    {
                        _dataObjects.AddOrUpdate(o);
                    }

                    GetElementStructureAsync(children);
                });

            _cleanUp.Add(disposable);
        }

        public Task<IEnumerable<IDataObject>> GetElementStructureAsync(IDataObject parent)
        {
            var children = _childrenFilter.ChildrenFilter(parent).ToList();
            if (!children.Any())
                return Task.FromResult<IEnumerable<IDataObject>>(Array.Empty<IDataObject>());

            return GetChildrenAsync(children);
        }

        public void Dispose()
        {
            _cleanUp.Dispose();
        }

        private async Task<IEnumerable<IDataObject>> GetChildrenAsync(IEnumerable<Guid> ids)
        {
            var result = new List<IDataObject>();
            var objects = await _repository.GetObjectsAsync(ids, CancellationToken.None);
            var collection = objects.ToList();
            result.AddRange(collection);
            foreach (var child in collection)
            {
                var childrenIds = _childrenFilter.ChildrenFilter(child).ToList();
                if (!childrenIds.Any())
                    continue;

                var children = await GetChildrenAsync(childrenIds);
                result.AddRange(children);
            }

            return result;
        }

        private void GetElementStructureAsync(IEnumerable<Guid> ids)
        {
            var disposable = _repository.SubscribeObjects(ids)
                .Where(x => _repository.GetCurrentAccess(x.Id, _repository.GetCurrentPerson().Id).HasFlag(AccessLevel.View))
                .Where(x => _childrenFilter.Filter(x))
                .SubscribeOnDispatcher(DispatcherPriority.Background)
                .Subscribe(o => { _dataObjects.AddOrUpdate(o); });
            _cleanUp.Add(disposable);
        }

        public IType GetType(int typeId)
        {
            return _repository.GetType(typeId);
        }
    }
}