/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using System.Reactive.Subjects;
using Ascon.Pilot.SDK;
using DynamicData;
using DynamicData.Binding;
using DynamicData.Kernel;
using ProjectCloneWizard.Common;
using ProjectCloneWizard.Data;


namespace ProjectCloneWizard.ObjectsTree
{
    public class NodeViewModel : AbstractNotifyPropertyChanged, IDisposable
    {
        private readonly Node<IDataObject, Guid> _node;
        private ReadOnlyObservableCollection<NodeViewModel> _children;
        private bool _isExpanded;
        private readonly IDisposable _cleanUp;
        private bool _isChecked = true;

        public NodeViewModel(Node<IDataObject, Guid> node, IDataObjectService dataObjectService, bool isExpand, NodeViewModel parent = null)
        {
            _node = node;
            Parent = parent;

            if (Parent.HasValue)
                IsChecked = Parent.Value.IsChecked;

            //Документы не выбираем изначально
            if (!node.Item.Type.Children
                .Select(dataObjectService.GetType).Any(t => !t.IsDeleted && !t.IsService) && node.Item.Type.HasFiles && !node.Item.Type.IsEcmDocument())
                _isChecked = false;
            else
                IsChecked = true;

            var childrenLoader = new Lazy<IDisposable>(() => node.Children.Connect()
                    .Transform(x => new NodeViewModel(x, dataObjectService, isExpand, this))
                    .Sort(new NodeViewModelComparer())
                    .Bind(out _children)
                    .DisposeMany()
                    .Subscribe());

            var shouldExpand = node.IsRoot
                ? Observable.Return(true)
                : Parent.Value.WhenValueChanged(x => x.IsExpanded);

            if (isExpand)
                shouldExpand = Observable.Return(true);

            var expander = shouldExpand
                    .Where(isExpanded => isExpanded)
                    .Take(1)
                    .Subscribe(x =>
                    {
                        dataObjectService.GetElementStructureAsync(node.Item.Id, () => { });
                        //force lazy loading
                        var value = childrenLoader.Value;
                    });

            _cleanUp = Disposable.Create(() =>
            {
                expander.Dispose();
                if (childrenLoader.IsValueCreated)
                    childrenLoader.Value.Dispose();
            });
        }

        public Guid Id => _node.Item.Id;
        public string Title => _node.Item.DisplayName;
        public byte[] Icon => _node.Item.Type.SvgIcon;
        public int Sort => _node.Item.Type.Sort;
        public Optional<NodeViewModel> Parent { get; }
        public ReadOnlyObservableCollection<NodeViewModel> Children => _children;
        public IType Type => _node.Item.Type;
        public IDictionary<string, object> Attributes => _node.Item.Attributes;
        public ReadOnlyCollection<IAccessRecord> Access => _node.Item.Access2;
        public bool IsSecret => _node.Item.IsSecret;
        public IDataObject Source => _node.Item;

        public bool IsExpanded
        {
            get => _isExpanded;
            set => SetAndRaise(ref _isExpanded, value);
        }

        private readonly Subject<bool> _isCheckedSubject = new Subject<bool>();
        public IObservable<bool> IsCheckedObservable => _isCheckedSubject;

        public bool IsChecked
        {
            get { return _isChecked; }
            set
            {
                if (value && Parent.HasValue)
                    Parent.Value.IsChecked = true;

                if (!value && Children != null)
                {
                    foreach (var child in Children)
                        child.IsChecked = false;
                }

                SetAndRaise(ref _isChecked, value);
                _isCheckedSubject.OnNext(value);
            }
        }

        public void Dispose()
        {
            _cleanUp.Dispose();
        }
    }
}
