/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Toolbar;
using DynamicData;
using ProjectCloneWizard.Common;

namespace ProjectCloneWizard
{
    [Export(typeof(IToolbar<ObjectsViewContext>))]
    public class ProjectCloneToolbar : ToolbarItemSubmenuHandler, IToolbar<ObjectsViewContext>
    {
        private readonly ITabServiceProvider _tabServiceProvider;
        private readonly IProjectCloneCreator _projectCloneCreator;
        private readonly ICopyStructureService _copyStructureService;
        private readonly IIconsProvider _iconsProvider;
        private IDataObject _selected;

        [ImportingConstructor]
        public ProjectCloneToolbar(IPilotDialogService pilotDialogService, ITabServiceProvider tabServiceProvider, 
            IProjectCloneCreator projectCloneCreator, ICopyStructureService copyStructureService, IIconsProvider iconsProvider)
        {
            _tabServiceProvider = tabServiceProvider;
            _projectCloneCreator = projectCloneCreator;
            _copyStructureService = copyStructureService;
            _iconsProvider = iconsProvider;
        }

        public void Build(IToolbarBuilder builder, ObjectsViewContext context)
        {
            var objects = context.SelectedObjects.ToList();
            if (objects.Count != 1)
                return;

            _selected = objects.First();

            AddMenuCreate(builder);
            AddMenuCopy(builder);
            AddMenuPaste(builder, context);
        }

        public async void OnToolbarItemClick(string name, ObjectsViewContext context)
        {
            if (name == Const.COMMAND_UPDATE_NAME || name == Const.COMMAND_NAME)
            {
                var selected = context.SelectedObjects.First();
                _projectCloneCreator.ShowWizardDialog(selected);
            }

            if (name == Const.COMMAND_COPY_STRUCTURE_HERE_NAME)
            {
                var selected = context.SelectedObjects.First();
                var idToShow = await _copyStructureService.CopyStructure(selected);
                _tabServiceProvider.ShowElement(idToShow);
            }

            if (name == Const.COMMAND_COPY_STRUCTURE_NAME)
            {
                var selected = context.SelectedObjects.FirstOrDefault();
                _copyStructureService.SetElementToCopy(selected);
            }

            if (name == Const.COMMAND_PASTE_STRUCTURE_NAME)
            {
                var copied = _copyStructureService.GetElementToCopy();
                var selected = context.SelectedObjects.First();
                var idToShow = await _copyStructureService.CopyStructure(copied, selected);
                _tabServiceProvider.ShowElement(idToShow);
            }
        }

        public override void OnSubmenuRequested(IToolbarBuilder builder)
        {
            var itemNames = builder.ItemNames.ToList();
            var index = itemNames.Count;

            var smartFolderIndex = itemNames.IndexOf("miCreateSmartfoldertype");
            if (smartFolderIndex != -1)
                index = smartFolderIndex == 0 ? itemNames.Count - 1 : smartFolderIndex - 1;

            if (_selected.Type.IsProject)
            {
                builder.AddButtonItem(Const.COMMAND_UPDATE_NAME, index)
                       .WithHeader(Const.COMMAND_UPDATE_TITLE)
                       .WithIcon(IconLoader.GetIcon(_projectCloneCreator.Icon));
            }
            else
            {
                builder.AddButtonItem(Const.COMMAND_NAME, index)
                       .WithHeader(Const.COMMAND_TITLE)
                       .WithIcon(IconLoader.GetIcon(_projectCloneCreator.Icon));
            }
        }

        private void AddMenuCreate(IToolbarBuilder builder)
        {
            var menu = builder.ItemNames.ToList();
            if (!menu.Contains("miCreate"))
                return;

            builder.ReplaceMenuButtonItem("miCreate").WithMenu(this);
        }

        private void AddMenuCopy(IToolbarBuilder builder)
        {
            var index = builder.ItemNames.IndexOf("miCopy");
            if (index == -1)
                return;

            builder.AddButtonItem(Const.COMMAND_COPY_STRUCTURE_HERE_NAME, ++index)
                .WithHeader(Const.COMMAND_COPY_STRUCTURE_HERE_TITLE)
                .WithIcon(_iconsProvider.GetCopyStructureHereIcon());

            builder.AddButtonItem(Const.COMMAND_COPY_STRUCTURE_NAME, ++index)
                .WithHeader(Const.COMMAND_COPY_STRUCTURE_TITLE)
                .WithIcon(_iconsProvider.GetCopyStructureIcon());
        }

        private void AddMenuPaste(IToolbarBuilder builder, ObjectsViewContext context)
        {
            if (!context.IsContext)
                return;

            var index = builder.ItemNames.Count();
            builder.AddButtonItem(Const.COMMAND_PASTE_STRUCTURE_NAME, index)
                .WithHeader(Const.COMMAND_PASTE_STRUCTURE_TITLE)
                .WithIcon(_iconsProvider.GetPasteStructureIcon())
                .WithIsEnabled(_copyStructureService.GetElementToCopy() != null);
        }
    }
}
