/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Controls.ObjectCardView;
using Ascon.Pilot.SDK.Extensions;
using ProjectCloneWizard.Common;
using ProjectCloneWizard.ObjectsTree;

namespace ProjectCloneWizard.ViewModels
{
    class ProcessPageViewModel : PropertyChangedBase, IWizardNavigationService
    {
        protected readonly CreationInfo _creationInfo;
        protected readonly CancellationTokenSource _cts = new CancellationTokenSource();
        protected readonly Dictionary<Guid, Guid> _idMap = new Dictionary<Guid, Guid>();
        protected readonly IObjectsRepository _repository;
        private readonly IObjectModifier _modifier;
        private readonly IFileProvider _fileProvider;
        private readonly int _currentPersonId;
        private bool _canGoNext = false;
        private string _progress;

        public ProcessPageViewModel(CreationInfo creationInfo, IObjectModifier modifier, IObjectsRepository repository, IFileProvider fileProvider)
        {
            _creationInfo = creationInfo;
            _modifier = modifier;
            _repository = repository;
            _fileProvider = fileProvider;
            _currentPersonId = repository.GetCurrentPerson().Id;
        }

        public string Progress
        {
            get { return _progress; }
            set
            {
                _progress = value;
                NotifyOfPropertyChanged(() => Progress);
            }
        }

        public async Task Start()
        {
            //Дадим команду визарду обновить состояние
            UpdateState?.Invoke(this, EventArgs.Empty);

            try
            {
                Progress = "Подготовка к копированию шаблона";
                var collection = new List<IDataObject>();
                Progress = "Загрузка всех элементов шаблона";
                await LoadCore(new[] { _creationInfo.Template.Id }, collection);

                Progress = "Создание корневого элемента";
                await Process(collection);

                _cts.Token.ThrowIfCancellationRequested();
                _modifier.Apply();
            }
            catch (OperationCanceledException)
            {
                _modifier.Clear();
            }
            catch (Exception e)
            {
                //Результат failed
                _creationInfo.Exception = e;
                _modifier.Clear();
            }
            finally
            {
                //Дадим команду визарду перейти на следующую страницу
                _canGoNext = true;
                NavigateNext?.Invoke(this, EventArgs.Empty);
            }
        }

        public bool CanGoNext()
        {
            return _canGoNext;
        }

        public string NextButtonCaption()
        {
            return null;
        }

        public event EventHandler UpdateState;
        public event EventHandler NavigateNext;
        public bool CanGoBack()
        {
            return _canGoNext;
        }

        public void Cancel()
        {
            _cts.Cancel();
        }

        protected virtual Task<bool> Process(IEnumerable<IDataObject> templateObjects)
        {
            return Task<bool>.Factory.StartNew(() =>
            {
                //Create project
                _creationInfo.ObjectIdToShow = CreateProjectClone();

                var objectList = new List<NodeViewModel>();
                FillNodeList(_creationInfo.ObjectsNodes, objectList);

                var storageList = new List<NodeViewModel>();
                FillNodeList(_creationInfo.StorageNodes, storageList);

                foreach (var dataObject in templateObjects)
                {
                    _cts.Token.ThrowIfCancellationRequested();

                    if (dataObject.Id == _creationInfo.Template.Id)
                        continue;

                    //если есть в списке и установлен IsChecked создаем
                    var node = objectList.FirstOrDefault(n => n.Id == dataObject.Id);
                    if (node != null && node.IsChecked)
                        CreateObjectClone(dataObject, _creationInfo.CopyAccessForObjects);

                    var storageNode = storageList.FirstOrDefault(n => n.Id == dataObject.Id);
                    if (storageNode != null && storageNode.IsChecked)
                    {
                        if (storageNode.Type.IsEcmDocument())
                            continue;

                        CreateObjectClone(dataObject, _creationInfo.CopyAccessForFiles);
                    }
                }

                return true;
            }, _cts.Token);
        }

        protected async Task LoadCore(IEnumerable<Guid> ids, ICollection<IDataObject> destination)
        {
            var nodes = await _repository.GetObjectsAsync(ids, _cts.Token);
            foreach (var node in nodes)
            {
                destination.Add(node);
                if (node.Children.Any())
                    await LoadCore(node.Children, destination);
            }
        }

        private Guid CreateProjectClone()
        {
            var typeTuple = GetTupleOfDeletedAndNotDeletedType(_creationInfo.Template.Type.Name);
            CheckTypeTuple(typeTuple);
            var newId = Guid.NewGuid();
            var builder = _modifier.CreateById(newId, _creationInfo.Parent.Id, typeTuple.Item2)
                .SetCreator(_currentPersonId);

            if (_creationInfo.Template.IsSecret)
                builder.MakeSecret();

            if (_creationInfo.CopyAccessForObjects)
                SetAccess(_creationInfo.Template.Access2, builder);

            foreach (var attribute in _creationInfo.Attributes)
            {
                if (builder.DataObject.Type.Attributes.Any(x => x.Name.Equals(attribute.Key)))
                    SetAttribute(attribute.Key, attribute.Value.Value, builder);
            }

            _idMap.Add(_creationInfo.Template.Id, newId);
            return newId;
        }

        protected bool CreateObjectClone(IDataObject dataObject, bool copyAccess)
        {
            Progress = $"Создание нового элемента {dataObject.DisplayName}";

            var parentId = _idMap.FirstOrDefault(k => k.Key == dataObject.ParentId);
            if (parentId.Value == Guid.Empty && parentId.Key == Guid.Empty)
                return false;

            var nodeTypeTuple = GetTupleOfDeletedAndNotDeletedType(dataObject.Type.Name);
            CheckTypeTuple(nodeTypeTuple);

            var newObjectId = Guid.NewGuid();
            var nodeBuilder = _modifier.CreateById(newObjectId, parentId.Value, nodeTypeTuple.Item2)
                .SetCreator(_currentPersonId);

            if (dataObject.IsSecret)
                nodeBuilder.MakeSecret();

            if (copyAccess)
                SetAccess(dataObject.Access2, nodeBuilder);

            foreach (var attribute in dataObject.Attributes)
            {
                SetAttribute(attribute.Key, attribute.Value, nodeBuilder);
            }

            //Копирование файлов
            foreach (var file in dataObject.Files)
            {
                if (file.Name == "PilotDigitalSignature")
                    continue;

                using (var stream = _fileProvider.OpenRead(file))
                {
                    nodeBuilder.AddFile(file.Name, stream, file.Created, file.Accessed, file.Modified);
                }
            }

            //Сохраним соответствие идентификаторов
            _idMap[dataObject.Id] = newObjectId;
            return true;
        }

        private static void SetAccess(IEnumerable<IAccessRecord> accessRecords, IObjectBuilder nodeBuilder)
        {
            foreach (var record in accessRecords)
            {
                if (record.Access.IsInherited)
                    continue;

                var accessValue = record.Access;
                nodeBuilder.AddAccessRecords(record.OrgUnitId, accessValue.AccessLevel, accessValue.ValidThrough, accessValue.Inheritance, accessValue.Type, accessValue.TypeIds);
            }
        }

        private void SetAttribute(string key, object value, IObjectBuilder builder)
        {
            var dValue = DValue.GetDValue(value);
            if (dValue.StrValue != null)
                builder.SetAttribute(key, dValue.StrValue);
            if (dValue.IntValue != null)
                builder.SetAttribute(key, dValue.IntValue.Value);
            if (dValue.DoubleValue != null)
                builder.SetAttribute(key, dValue.DoubleValue.Value);
            if (dValue.DateValue != null)
                builder.SetAttribute(key, dValue.DateValue.Value);
            if (dValue.DecimalValue != null)
                builder.SetAttribute(key, dValue.DecimalValue.Value);
            if (dValue.GuidValue != null)
                builder.SetAttribute(key, dValue.GuidValue.Value);
            if (dValue.ArrayIntValue != null)
                builder.SetAttribute(key, dValue.ArrayIntValue);
            if (dValue.ArrayValue != null)
                builder.SetAttribute(key, dValue.ArrayValue);
            if (dValue.ArrayValue != null)
                builder.SetAttributeAsObject(key, dValue.ArrayValue);
        }

        protected void FillNodeList(ICollection<NodeViewModel> nodes, ICollection<NodeViewModel> list)
        {
            if (nodes == null)
                return;

            foreach (var checkedNode in nodes)
            {
                list.Add(checkedNode);
                FillNodeList(checkedNode.Children, list);
            }
        }

        private Tuple<IType, IType> GetTupleOfDeletedAndNotDeletedType(string typeName)
        {
            var types = _repository.GetTypes().Where(t => t.Name == typeName).ToList();
            return Tuple.Create(types.FirstOrDefault(t => t.IsDeleted), types.FirstOrDefault(t => !t.IsDeleted));
        }

        private void CheckTypeTuple(Tuple<IType, IType> isDeletedAndIsNotDeleted)
        {
            if (isDeletedAndIsNotDeleted.Item1 != null & isDeletedAndIsNotDeleted.Item2 == null)
                throw new Exception($"Невозможно создать объект с удаленным типом \"{isDeletedAndIsNotDeleted.Item1.Title}\". Удалите объект с этим типом в шаблоне и повторите попытку.");
        }
    }
}
