/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reactive.Linq;
using System.Text;
using System.Threading.Tasks;
using Ascon.Pilot.SDK.Controls.Commands;
using DynamicData.Binding;
using ProjectCloneWizard.Common;
using ProjectCloneWizard.ObjectsTree;
using ProjectCloneWizard.Wizard;

namespace ProjectCloneWizard.ViewModels
{
    class ProjectCompositionPageViewModel : PropertyChangedBase, IWizardCaptionProvider, IDisposable
    {
        private readonly CreationInfo _creationInfo;
        private readonly IDataObjectServiceFactory _serviceFactory;
        private ObjectsTreeViewModel _objectsTreeViewModel;
        private Guid _currentTemplateId;
        private readonly DelegateCommand _copyAccessRightsCommand;
        private ToolbarToggleButtonItem _copyToolbarItem;
        private IDisposable _subscription;

        public ProjectCompositionPageViewModel(CreationInfo creationInfo, IDataObjectServiceFactory serviceFactory)
        {
            _creationInfo = creationInfo;
            _serviceFactory = serviceFactory;
            _copyAccessRightsCommand = new DelegateCommand(CopyAccess);
        }

        public ObjectsTreeViewModel ObjectsTreeViewModel
        {
            get => _objectsTreeViewModel;
            set
            {
                _objectsTreeViewModel = value;
                NotifyOfPropertyChanged(() => ObjectsTreeViewModel);
            }
        }

        public void Load()
        {
            if (_currentTemplateId == _creationInfo.Template.Id)
                return;

            _currentTemplateId = _creationInfo.Template.Id;

            ObjectsTreeViewModel = new ObjectsTreeViewModel(_serviceFactory.GetService(), true);
            _subscription = ObjectsTreeViewModel.CheckedList.Subscribe(list =>
            {
                _creationInfo.ObjectsNodes = list;
            });
            
            FillToolbar();
            _objectsTreeViewModel.Load(_creationInfo.Template.Id);
        }

        public string GetTitle()
        {
            return "Шаг 2 из 4. Выберите необходимые элементы состава";
        }

        public string GetTemplateName()
        {
            return _creationInfo.Template.DisplayName;
        }

        private void FillToolbar()
        {
            _objectsTreeViewModel.ToolbarItemsSource.Add(new ToolbarButtonItem("CollapseAll")
            {
                Header = "Свернуть все",
                Icon = Icons.Instance.CollapseAllIcon,
                Command = _objectsTreeViewModel.CollapseAllCommand,
            });
            _objectsTreeViewModel.ToolbarItemsSource.Add(new ToolbarButtonItem("ExpandAll")
            {
                Header = "Развернуть все",
                Icon = Icons.Instance.ExpandAllIcon,
                Command = _objectsTreeViewModel.ExpandAllCommand,
            });
            _objectsTreeViewModel.ToolbarItemsSource.Add(new ToolbarButtonItem("CheckedAll")
            {
                Header = "Выбрать все",
                Icon = Icons.Instance.CheckAllIcon,
                Command = _objectsTreeViewModel.CheckAllCommand,
            });
            _objectsTreeViewModel.ToolbarItemsSource.Add(new ToolbarButtonItem("UncheckedAll")
            {
                Header = "Снять все",
                Icon = Icons.Instance.UncheckAllIcon,
                Command = _objectsTreeViewModel.UncheckAllCommand,
            });
            _copyToolbarItem = new ToolbarToggleButtonItem("CopyAccess")
            {
                Header = "Скопировать права доступа для выбранных элементов",
                Icon = Icons.Instance.UncheckAllIcon,
                IsChecked = false,
                Command = _copyAccessRightsCommand
            };
            _objectsTreeViewModel.ToolbarItemsSource.Add(_copyToolbarItem);
        }

        private void CopyAccess()
        {
            _copyToolbarItem.IsChecked = !_copyToolbarItem.IsChecked;
            _copyToolbarItem.Icon = _copyToolbarItem.IsChecked ? Icons.Instance.CheckAllIcon : Icons.Instance.UncheckAllIcon;
            _creationInfo.CopyAccessForObjects = _copyToolbarItem.IsChecked;
        }

        public void Dispose()
        {
            _objectsTreeViewModel?.Dispose();
            _subscription?.Dispose();
        }
    }
}
