/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Reactive.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Threading;
using Ascon.Pilot.SDK;
using IDataObject = Ascon.Pilot.SDK.IDataObject;

namespace ShortcutRemover
{
    [Export(typeof(IObjectChangeProcessor))]
    public class ChangeProcessor : IObjectChangeProcessor
    {
        private readonly IObjectsRepository _repository;
        private readonly IObjectModifier _modifier;
        private readonly ISearchService _searchService;
        private readonly int _shortcutTypeId;

        [ImportingConstructor]
        public ChangeProcessor(IObjectsRepository repository, IObjectModifier modifier, ISearchService searchService)
        {
            _repository = repository;
            _modifier = modifier;
            _searchService = searchService;
            _shortcutTypeId = _repository.GetTypes().First(x => x.Name == SystemTypeNames.SHORTCUT).Id;
        }

        public bool ProcessChanges(IEnumerable<DataObjectChange> changes, IObjectModifier modifier)
        {
            foreach (var change in changes)
            {
                if (change.Old != null && 
                    change.Old.ObjectStateInfo.State != ObjectState.InRecycleBin &&
                    change.New.ObjectStateInfo.State == ObjectState.InRecycleBin)
                {
                    _ = HandleObjectRemove(change.New.Id);
                }
            }
            return true;
        }

        private async Task HandleObjectRemove(Guid objectId)
        {
            var builder = _searchService.GetEmptyQueryBuilder();
            builder.Must(ObjectFields.ObjectState.Be(ObjectState.Alive));
            builder.Must(ObjectFields.TypeId.Be(_shortcutTypeId));
            builder.Must(AttributeFields.String(SystemAttributeNames.SHORTCUT_OBJECT_ID).Be(objectId.ToString()));
            var searchResult = await _searchService.Search(builder).FirstAsync(x => x.Kind == SearchResultKind.Remote);
            var shortcuts = await GetObjectsAsync(searchResult.Result, CancellationToken.None);
            
            if(!shortcuts.Any())
                return;

            var currentPerson = _repository.GetCurrentPerson();
            var removableShortcuts = shortcuts
                .Where(s => _repository.GetCurrentAccess(s.Id, currentPerson.Id).HasFlag(AccessLevel.Edit))
                .ToList();


            var obj = (await GetObjectsAsync(new[] { objectId }, CancellationToken.None)).First();
            
            var sb = new StringBuilder();
            sb.AppendLine($@"Найдено {shortcuts.Count} ярлыков к элементу ""{obj.DisplayName}"". ");
            if(removableShortcuts.Count != shortcuts.Count)
                sb.AppendLine($"Из них доступно по правам доступа {removableShortcuts.Count}. ");

            if (!removableShortcuts.Any())
            {
                MessageBox.Show(sb.ToString(), "Удаление элемента", MessageBoxButton.OK);
                return;
            }

            sb.AppendLine($"Удалить их?");

            if (MessageBox.Show(sb.ToString(), "Удаление элемента", MessageBoxButton.YesNo) == MessageBoxResult.Yes)
            {
                foreach (var shortcut in removableShortcuts)
                {
                    _modifier.Delete(shortcut);
                }
                _modifier.Apply();
            }
        }

        /// <summary>
        /// Returns objects by identifiers as an asynchronous operation
        /// </summary>
        /// <param name="ids">Object identifiers to be loaded</param>
        /// <param name="ct">CancellationToken to cancel objects loading. The OperationCanceledException will be thrown.</param>
        /// <returns>The task object representing the asynchronous operation</returns>
        public Task<IReadOnlyList<IDataObject>> GetObjectsAsync(IEnumerable<Guid> ids, CancellationToken ct)
        {
            // Creates an observable that fires notification when cancelling CancellationToken
            var cancel = Observable.Create<IDataObject>(o => ct.Register(o.OnCompleted));

            var loading = ids.ToList();
            var observableList = _repository
                .SubscribeObjects(loading)                          // Subscribing on interested objects
                .TakeUntil(cancel)                                  // Stopping subscription on cancel
                .ObserveOnDispatcher(DispatcherPriority.Background)
                .Where(o => o.State == DataState.Loaded)   // Filtering "NoData" notifications
                .Distinct(o => o.Id)                       // Filtering already emitted notifications
                .Take(loading.Count)                                // Wait for all objects to be loaded
                .ToList();

            return Task<IReadOnlyList<IDataObject>>.Factory.StartNew(() =>
            {
                var lazy = observableList.Wait();
                // Forces the "lazy" enumerable to be immediately iterated in background thread
                return lazy.ToList();
            }, ct);
        }
    }
}
