/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Reactive.Disposables;
using Ascon.Pilot.SDK;
using DynamicData;
using DynamicData.Binding;

namespace ShowObjectsTree.DynamicData
{
    public class ObjectsTreeViewModel : AbstractNotifyPropertyChanged, IDisposable
    {
        private readonly ReadOnlyObservableCollection<NodeViewModel> _nodes;
        private readonly DataObjectService _dataObjectService;
        private readonly IDisposable _cleanUp;
        
        public ObjectsTreeViewModel(IObjectsRepository repository)
        {
            var compositeDisposable = new CompositeDisposable();

            _dataObjectService = new DataObjectService(repository);
            compositeDisposable.Add(_dataObjectService);

            var disposable = _dataObjectService.DataObjects.Connect()
                .TransformToTree(o => o.ParentId)
                .Transform(node => new NodeViewModel(node, _dataObjectService))
                .Sort(new NodeViewModelComparer())
                .Bind(out _nodes)
                .DisposeMany()
                .Subscribe();
            compositeDisposable.Add(disposable);

            _cleanUp = compositeDisposable;
        }

        public ReadOnlyObservableCollection<NodeViewModel> Nodes => _nodes;

        public void Load(IEnumerable<Guid> ids)
        {
            _dataObjectService.Load(ids);
        }

        public void Dispose()
        {
            _cleanUp.Dispose();
        }
    }
}
