/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using Ascon.Pilot.Common;
using Ascon.Pilot.ServerExtensions.SDK;
using Prometheus;

namespace PilotServer.PrometheusMetrics
{
    [Export(typeof(IMetricsApi))]
    public class MetricsApi : IMetricsApi
    {
        private const string LICENSE_TYPE_LABEL_NAME = "licenseType";
        // в данный список могут быть добавлены коды модулей расширения
        private static readonly IReadOnlyDictionary<int, string> ProductLabels = new Dictionary<int, string>()
        {
            { 100, "pilot_ice" },
            { 103, "pilot_ice_enterprise" },
            { 101, "pilot_ecm" },
            { 90 , "pilot_bim" }
        };

        private static readonly Gauge ServerConnectionCount = Metrics
            .CreateGauge("pilot_server_connection_count", "Общее количество подключений к Pilot-Server (любыми клиенсткими приложениями)");

        private static readonly Gauge UserAuthFailedCount = Metrics
            .CreateGauge("pilot_server_user_auth_failed_count", "Количество последовательных неуспешных попыток аутентификации пользователя БД");

        private static readonly Gauge AdministratorAuthFailedCount = Metrics
            .CreateGauge("pilot_server_admin_auth_failed_count", "Количество последовательных неуспешных попыток аутентификации администратора Pilot-Server");

        private static readonly Gauge ChangesetsQueueLength = Metrics
            .CreateGauge("pilot_server_changesets_queue_length", "Количество изменений в очереди на обработку");

        private static readonly Gauge ChangesetsErrors = Metrics
            .CreateGauge("pilot_server_changesets_errors", "Количество не принятых сервером изменений");

        private static readonly Gauge ConsumedLicenses = Metrics
            .CreateGauge("pilot_server_consumed_licenses", "Количество взятых лицензий по продукту", LICENSE_TYPE_LABEL_NAME);

        private static readonly Gauge ConsumeLicenseErrorsCount = Metrics
            .CreateGauge("pilot_server_consume_license_errors", "Количество неуспешных попыток взятия лицензии", LICENSE_TYPE_LABEL_NAME);

        private static readonly Gauge MessagesIndexTasksCount = Metrics.CreateGauge("pilot_messages_index_tasks_count",
            "Колличество задач на индексацию сообщений в очереди");

        private static readonly Gauge ChangesetsIndexTasksCount = Metrics.CreateGauge("pilot_changesets_index_tasks_count",
            "Колличество задач на индексацию изменений в очереди");

        private static readonly Gauge RedisConnectionLostCount =
            Metrics.CreateGauge("pilot_redis_connection_lost_count", " Число потерть соединения с редис");

        private static bool _started;
        private readonly ConcurrentDictionary<string, int> _failedUserAuthCounts = new ConcurrentDictionary<string, int>();
        private readonly ConcurrentDictionary<string, int> _failedAdminAuthCounts = new ConcurrentDictionary<string, int>();
        private readonly ConcurrentDictionary<Guid, bool> _changesets = new ConcurrentDictionary<Guid, bool>();
        private readonly ConcurrentDictionary<int, ConcurrentHashSet<string>> _failedToConsumeLicenseByProduct = new ConcurrentDictionary<int, ConcurrentHashSet<string>>();
        private readonly IndexQueueState _messagesIndexQueueState = new IndexQueueState();
        private readonly IndexQueueState _changesetsIndexQueueState = new IndexQueueState();
        private  const string STATE_CHHANGED_REASON_REDIS_CONNECTION_LOST = "Redis connection lost";


        [ImportingConstructor]
        public MetricsApi(IMetricsEvents events)
        {
            if(_started)
                return;

            var server = new MetricServer(1234);
            server.Start();

            // мониторинг системных метрик поддерживается только в Pilot-Server на ОС Linux (.NET 6)
#if NETSTANDARD2_1
            Prometheus.DotNetRuntime.DotNetRuntimeStatsBuilder
                .Customize()
                .WithExceptionStats()
                .WithThreadPoolStats()
                .WithSocketStats()
                .StartCollecting();
#endif

            events.ConnectionCountChanged += (o, e) => ServerConnectionCount.Set(e.ConnectionCount);
            events.OnUserAuth += OnUserAuth;
            events.OnAdministratorAuth += OnAdministratorAuth;
            events.OnChangeset += OnChangeset;
            events.LicenseConsumed += OnLicenseConsumed;
            events.LicenseReleased += OnLicenseReleased;
            events.LicenseConsumeError += OnLicenseConsumeError;
            events.DatabaseStateChanged += OnDatabaseStateChanged;
            events.MessagesQueuedForIndex += OnMessagesQueuedForIndex;
            events.MessagesIndexed += OnMessagesIndexed;
            events.ChangesetsQueuedForIndex += OnChangesetsQueuedForIndexChanged;
            events.ChangesetsIndexed += OnChangesetsIndexed;

            _started = true;
        }

        private void OnChangesetsIndexed(object sender, ChangesetIndexedEventArgs e)
        {
            _changesetsIndexQueueState.OnIndexed(e.ChangedChangesCount);
            ChangesetsIndexTasksCount.Set(e.QueuedTasksCount);
        }

        private void OnChangesetsQueuedForIndexChanged(object sender, ChangesetQueuedEventArgs e)
        {
            _changesetsIndexQueueState.OnQueued(e.ChangedChangesCount);
            ChangesetsIndexTasksCount.Set(e.QueuedTasksCount);
        }

        private void OnMessagesIndexed(object sender, MessagesIndexedEventArgs e)
        {
            _messagesIndexQueueState.OnIndexed(e.MessagesCount);
            MessagesIndexTasksCount.Set(e.QueuedTasksCount);
        }

        private void OnMessagesQueuedForIndex(object sender, MessagesQueuedForIndexEventArgs e)
        {
            _messagesIndexQueueState.OnQueued(e.MessagesCount);
            MessagesIndexTasksCount.Set(e.QueuedTasksCount);
        }

        private void OnDatabaseStateChanged(object sender, DatabaseStateChangedEventArgs e)
        {
            if (e.DatabaseState == DatabaseState.Broken && e.Reason == STATE_CHHANGED_REASON_REDIS_CONNECTION_LOST)
            {
                RedisConnectionLostCount.Inc();
            }
        }

        private void OnUserAuth(object sender, AuthEventArgs e)
        {
            if (e.IsSuccess)
                _failedUserAuthCounts.TryRemove(e.Login, out _);
            else
                _failedUserAuthCounts.AddOrUpdate(e.Login, _ => 1, (_, c) => c + 1);
            UserAuthFailedCount.Set(_failedUserAuthCounts.Values.DefaultIfEmpty().Max());
        }

        private void OnAdministratorAuth(object sender, AuthEventArgs e)
        {
            if (e.IsSuccess)
                _failedAdminAuthCounts.TryRemove(e.Login, out _);
            else
                _failedAdminAuthCounts.AddOrUpdate(e.Login, _ => 1, (_, c) => c + 1);
            AdministratorAuthFailedCount.Set(_failedAdminAuthCounts.Values.DefaultIfEmpty().Max());
        }

        private void OnChangeset(object sender, ChangesetEventArgs e)
        {
            if (e.Status == ChangesetStatus.Enqueued)
            {
                _changesets.TryAdd(e.ChangesetIdentity, true);
            }
            else
            {
                if(!_changesets.ContainsKey(e.ChangesetIdentity))
                    return;

                if(e.Status == ChangesetStatus.Error)
                    ChangesetsErrors.Inc();

                _changesets.TryRemove(e.ChangesetIdentity, out _);
            }
            ChangesetsQueueLength.Set(_changesets.Count);
        }

        private void OnLicenseConsumed(object sender, LicenseEventArgs e)
        {
            if(!ProductLabels.TryGetValue(e.LicenseType, out var label))
                return;

            ConsumedLicenses.WithLabels(label).Set(e.ConsumedCount);

            if (_failedToConsumeLicenseByProduct.TryGetValue(e.LicenseType, out var failedLogins) && failedLogins.TryRemove(e.Login))
                ConsumeLicenseErrorsCount.WithLabels(label).Set(failedLogins.Count());
        }
        
        private void OnLicenseReleased(object sender, LicenseEventArgs e)
        {
            if(!ProductLabels.TryGetValue(e.LicenseType, out var label))
                return;

            ConsumedLicenses.WithLabels(label).Set(e.ConsumedCount);
        }

        private void OnLicenseConsumeError(object sender, LicenseConsumeErrorEventArgs e)
        {
            if(!ProductLabels.TryGetValue(e.LicenseType, out var label))
                return;

            var failedLogins = _failedToConsumeLicenseByProduct.GetOrAdd(e.LicenseType, _ => new ConcurrentHashSet<string>());
            if(failedLogins.TryAdd(e.Login))
                ConsumeLicenseErrorsCount.WithLabels(label).Set(failedLogins.Count());
        }
    }
}