/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Threading.Tasks;
using Ascon.Pilot.Common;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.DataModifier;
using Ascon.Pilot.ServerExtensions.SDK;
using Newtonsoft.Json;

namespace ServerActivitySample
{
    [Export(typeof(IServerActivity))]
    public class UpdateDocumentStateOnSigning : IServerActivity
    {
        public string Name => "UpdateDocumentStateOnSigning";
        public const string StateAttributeName = "stateAttributeName";
        public const string SignedStateName = "signedStateName";
        public const string NotSignedStateName = "notSignedStateName";
        public const string StatesFrom = "statesFrom";

        public Task RunAsync(IModifierBase modifier, IModifierBackend backend, IServerActivityContext serverActivityContext, IAutomationEventContext automationEventContext)
        {
            var notifier = serverActivityContext.GetService<IAutomationNotifier>();

            var source = automationEventContext.Source;
            var isFullySigned = IsFullySigned(source);
            if(!serverActivityContext.Params.TryGetValue(StateAttributeName, out var stateAttributeName))
                throw new InvalidOperationException($"{StateAttributeName} parameter is not defined in UpdateDocumentStateOnSigning activity");
            if(!serverActivityContext.Params.TryGetValue(SignedStateName, out var signedStateName))
                throw new InvalidOperationException($"{SignedStateName} parameter is not defined in UpdateDocumentStateOnSigning activity");
            if(!serverActivityContext.Params.TryGetValue(NotSignedStateName, out var notSignedStateName))
                throw new InvalidOperationException($"{NotSignedStateName} parameter is not defined in UpdateDocumentStateOnSigning activity");
            serverActivityContext.Params.TryGetValue(StatesFrom, out var statesFrom);

            if (statesFrom != null && statesFrom is IEnumerable list)
            {
                var statesFromArray = JsonConvert.DeserializeObject<List<string>>(statesFrom.ToString()); 
                
                var currentState = backend.GetMetadata().UserStates
                    .FirstOrDefault(x => x.Id == source.Attributes[(string) stateAttributeName]);
                if (currentState == null || !statesFromArray.Contains(currentState.Name))
                    return Task.CompletedTask;
            }

            var signedState = backend.GetMetadata().UserStates.FirstOrDefault(x => x.Name == (string)signedStateName);
            if(signedState == null)
                throw new InvalidOperationException($"State {signedStateName} not found");

            var notSignedState = backend.GetMetadata().UserStates.FirstOrDefault(x => x.Name == (string)notSignedStateName);
            if(notSignedState == null)
                throw new InvalidOperationException($"State {NotSignedStateName} not found");

            var newStateId = isFullySigned ? signedState.Id : notSignedState.Id;
            if (!source.Attributes.TryGetValue((string)stateAttributeName, out var stateAttributeValue) || stateAttributeValue.GuidValue != newStateId)
            {
                modifier.EditObject(source).SetAttribute((string)stateAttributeName, newStateId);
                NotifySigned(notifier, backend, source);
            }

            return Task.CompletedTask;
        }

        public static bool IsFullySigned(INObject @object)
        {
            var document = GetDocument(@object);
            if (document == null || document.SignatureRequests.Count == 0)
                return false;

            var requestCount = document.SignatureRequests.Count;
            var signedRequest = document.SignatureRequests.Count(req => req.Signs.Any() && !req.IsAdvancementRequired);
            return signedRequest == requestCount;
        }

        private static INFile GetDocument(INObject @object)
        {
            var xps = @object.ActualFileSnapshot.Files.FirstOrDefault(f => FileExtensionHelper.IsXpsAlike(f.Name));
            if (xps != null)
                return xps;
            return @object.ActualFileSnapshot.Files.FirstOrDefault(f => FileExtensionHelper.IsPdfAlike(f.Name));
        }

        private void NotifySigned(IAutomationNotifier notifier, IModifierBackend backend, INObject @object)
        {
            var doc = GetDocument(@object);
            if (doc == null || doc.SignatureRequests.Count == 0)
                return;

            var positionsToNotify = new List<int>();
            foreach (var signature in doc.SignatureRequests.Where(x => x.Signs.Any() && !x.IsAdvancementRequired))
            {
                var relatedTaskId = signature.ObjectId;
                var relatedTask = backend.GetObject(relatedTaskId);
                if(relatedTask == null || !relatedTask.Attributes.TryGetValue(SystemTaskAttributes.EXECUTOR_POSITION, out var executorValue))
                    continue;

                if(executorValue != null && executorValue.ArrayIntValue != null)
                    positionsToNotify.AddRange(executorValue.ArrayIntValue);
            }
            
            var peopleToNotify = positionsToNotify.SelectMany(backend.GetPeopleOnPosition);
            foreach (var person in peopleToNotify)
            {
                notifier.Notify(person.Id, @object, "Документ согласован", Name);
            }
        }
    }
}
