﻿using Ascon.Pilot.Common.DataProtection;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.Server.Api;
using ServerApiFactory.Contracts;
using Spectre.Console;

const int PILOT_ICE_ENTERPRISE_CODE = 103; // Pilot-ICE Enteprise

var credentials = ConnectionCredentials.GetConnectionCredentials(
    AnsiConsole.Prompt(new TextPrompt<string>("Enter Pilot-Server address with database:")), 
    AnsiConsole.Prompt(new TextPrompt<string>("Enter login:")), 
    AnsiConsole.Prompt(new TextPrompt<string>("Enter password:").Secret()).ConvertToSecureString());

using var client = new HttpPilotClient(credentials.GetConnectionString());
client.Connect(false);
var auth = client.GetAuthenticationAsyncApi();
await auth.LoginWithIdentityAsync(credentials.DatabaseName, credentials.Username, credentials.ProtectedPassword, PILOT_ICE_ENTERPRISE_CODE, new SelfIdentity
{
    Id = "ServerApiFactory.CustomClient"
});

var table = new Table();
table.AddColumn("Author");
table.AddColumn("Text");

var api = client.GetCustomApi<ICustomServerContract, ICustomServerCallback>(new CustomServerCallback(PrintMessage));

var messages = await api.GetMessagesAsync();
messages.ForEach(PrintMessage);

await api.AddMessageAsync($"Hello! It's {DateTime.Now:t} now.");

await AnsiConsole.Live(table)
    .AutoClear(true)
    .Cropping(VerticalOverflowCropping.Top)
    .StartAsync(async ctx =>
    {
        while (true)
        {
            ctx.Refresh();
            await Task.Delay(500);
        }
    });

client.Disconnect();

void PrintMessage(Message message)
{
    table.AddRow(message.Author, message.Text);
}

class CustomServerCallback : ICustomServerCallback
{
    private readonly Action<Message> _onMessageAdded;

    public CustomServerCallback(Action<Message> onMessageAdded)
    {
        _onMessageAdded = onMessageAdded;
    }

    public void OnMessageAdded(Message message)
    {
        _onMessageAdded.Invoke(message);
    }
}