/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using Ascon.Pilot.DataClasses;

namespace Pilot.Xps.Domain.CertificateStorage
{
    public interface ICertificateStorage
    {
        X509Certificate2 GetCertificate2(INPerson person);
    }

    public class CertificateStorage : ICertificateStorage
    {
        private readonly IDictionary<string, string> _certificatesMap;
        private readonly IDictionary<string, X509Certificate2> _certificateCollection;

        public CertificateStorage(ICertificateStorageSettings settings)
        {
            _certificatesMap = settings.CertificatesMap;
            _certificateCollection = GetCertificatesCollection(settings.StoreLocation);
        }

        public X509Certificate2 GetCertificate2(INPerson person)
        {
            if (_certificateCollection.Count == 0)
                throw new Exception("Certificates for signing are not found or do not meet signing criteria.");

            var email = person.Email;
            if (_certificatesMap.ContainsKey(email))
            {
                email = _certificatesMap[email];
            }

            var certificate = _certificateCollection.TryGetValue(email, out var certificate2) ? certificate2 : null;
            if (certificate == null)
                throw new Exception("Certificate not found. Check user email.");

            return certificate;
        }

        private static IDictionary<string, X509Certificate2> GetCertificatesCollection(StoreLocation storeLocation)
        {
            var certificateCollection = GetCertificates(StoreName.My, storeLocation);
            certificateCollection = certificateCollection.Find(X509FindType.FindByTimeValid, DateTime.Now, true);
            certificateCollection = certificateCollection.Find(X509FindType.FindByKeyUsage, X509KeyUsageFlags.DigitalSignature, false);
            var map = new Dictionary<string, X509Certificate2>();
            foreach (var cer in certificateCollection.Cast<X509Certificate2>().Where(cer => cer.HasPrivateKey))
            {
                var email = cer.GetNameInfo(X509NameType.EmailName, false);
                if (email == null)
                    continue;

                map[email] = cer;
            }
            return map;
        }

        private static X509Certificate2Collection GetCertificates(StoreName storeName, StoreLocation storeLocation)
        {
            var store = new X509Store(storeName, storeLocation);
            store.Open(OpenFlags.OpenExistingOnly | OpenFlags.ReadOnly);
            var certificateCollection = store.Certificates;
            store.Close();

            return certificateCollection;
        }
    }
}
