/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Pilot.Xps.Domain.Print
{
    static class PagesPrintOrder
    {
        public static int[] Parse(string pageRange, int maxPage, int defaultPage)
        {
            if (string.IsNullOrEmpty(pageRange))
                return new[] { defaultPage };
            if (string.Equals(pageRange, "All", StringComparison.OrdinalIgnoreCase))
                return PagesPrintOrder.Parse("1-", maxPage);
            int[] parsedRange = null;
            try
            {
                parsedRange = PagesPrintOrder.Parse(pageRange, maxPage);
            }
            catch
            {
                return new int[0];
            }

            return parsedRange;
        }

        /// <summary>
        /// Парсит строку типа "1-2, 5-18" как диапазоны печати
        /// Диапазоны объединяются слева направо
        /// </summary>
        /// <param name="text">Строка</param>
        /// <param name="maxPage">Максимальная страница</param>
        /// <returns>Массив индексов страниц для печати</returns>
        public static int[] Parse(string text, int maxPage)
        {
            if (text == null || string.IsNullOrEmpty(text.Trim()))
                return new int[0];

            var pageRangeStrings = text.Split(',');
            var listOfParsedPageRanges = new List<int>();

            //Парсим строку в массивы чисел
            foreach (var pageRangeString in pageRangeStrings)
            {
                var array = pageRangeString.Split('-');
                if (array.Length == 1)
                {
                    var page = int.Parse(array.First());
                    if (page <= maxPage && page > 0)
                        listOfParsedPageRanges.Add(page);
                }
                else
                {
                    var first = array.First();
                    var last = array.Last();

                    var pageFrom = string.IsNullOrEmpty(first) ? 0 : int.Parse(first);
                    var pageTo = string.IsNullOrEmpty(last) ? maxPage : int.Parse(last);

                    var intArray = IntArray(pageFrom, pageTo);
                    foreach (var page in intArray)
                    {
                        // Не попадает в диапазон страниц
                        if (page > maxPage || page <= 0)
                            continue;

                        listOfParsedPageRanges.Add(page);
                    }
                }
            }

            return listOfParsedPageRanges.ToArray();
        }

        public static bool Validate(string text)
        {
            return Validate(text, Int32.MaxValue);
        }

        /// <summary>
        /// Проверяет является ли текст корректным диапазоном печати
        /// </summary>
        /// <param name="text">Выходной текст</param>
        /// <param name="maxPage">Максимальная страница</param>
        /// <returns>Результат проверки</returns>
        public static bool Validate(string text, int maxPage)
        {
            try
            {
                var pages = Parse(text, maxPage);

                if (pages == null || !pages.Any())
                    return false;

                return true;
            }
            catch (Exception)
            {
                return false;
            }
        }

        /// <summary>
        /// Возвращает массив чисел от from до to
        /// </summary>
        /// <param name="from">Нижняя граница</param>
        /// <param name="to">Верхняя граница</param>
        /// <returns>Результирующий массив</returns>
        public static int[] IntArray(int from, int to)
        {
            from = from > 0 ? from : 1;
            to = to > 0 ? to : 1;

            var count = Math.Abs(from - to) + 1;
            if (from == to)
            {
                return new int[] { from };
            }

            if (from < to)
            {
                return Enumerable.Range(from, count).ToArray();
            }

            return Enumerable.Range(to, count).Reverse().ToArray();
        }
    }
}
