/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Xml.Linq;
using Pilot.Xps.Domain.Labels;
using Pilot.Xps.Domain.Tools;
using ZXing;
using ZXing.Common;
using Path = System.Windows.Shapes.Path;

namespace Pilot.Xps.Domain.Render
{
    class BarcodeCreator : IBarCodeCreator
    {
        internal static BitMatrix Encode(string content)
        {
            var writer = new BarcodeWriter()
            {
                Format = BarcodeFormat.CODE_128,
                Options = new EncodingOptions { PureBarcode = true }

            };
            return writer.Encode(content);
        }


        public static Tuple<string, Uri, Stream, int> GetBarcodeInPathes(string content, string offsetX, string offsetY, string barcodeHeight, string scale, bool showText, bool textOnly)
        {
            var readable = new Tuple<string, Uri, Stream, int>("", null, null, 0);
            var result = new StringBuilder();
            var bitMatrix = Encode(content);
            result.AppendFormat(CultureInfo.InvariantCulture, "<Canvas Name=\"barcode\" RenderTransform=\"{2}, 0, 0, {2}, {0}, {1}\">", offsetX, offsetY, scale);

            if (!textOnly)
            {
                for (var i = 0; i < bitMatrix.Width; i++)
                {
                    if (bitMatrix[i, 0])
                    {
                        var currentLineEndX = GetEndX(bitMatrix, i);
                        result.AppendFormat(CultureInfo.InvariantCulture, "<Path Data=\"M {0},0 L {1},0 {1},{2} {0},{2} Z\" Fill=\"#000000\" />", i, currentLineEndX, barcodeHeight);
                        i = currentLineEndX;
                    }
                }
            }

            if (showText)
            {
                var readableBarCode = new TextBlock() { Text = content, Width = bitMatrix.Width, TextAlignment = TextAlignment.Center };

                if (textOnly)
                {
                    readableBarCode = new TextBlock() { Text = content, TextAlignment = TextAlignment.Center };
                    readable = XpsWriter.GenerateGlyphsFromTextBlock(readableBarCode);
                    result.AppendFormat(CultureInfo.InvariantCulture, "<Path Data=\"M 0,0 L {0},0 {0},{1} 0,{1} Z\" Opacity=\"0\"/>",
                        readableBarCode.ActualWidth, readableBarCode.ActualHeight);
                }
                else
                {
                    var barcodeH = Convert.ToDouble(barcodeHeight, CultureInfo.InvariantCulture);
                    Canvas.SetTop(readableBarCode, barcodeH);
                    readable = XpsWriter.GenerateGlyphsFromTextBlock(readableBarCode);
                    result.AppendFormat(CultureInfo.InvariantCulture, "<Path Data=\"M 0,0 L {0},0 {0},{1} 0,{1} Z\" Opacity=\"0\"/>",
                        bitMatrix.Width, readableBarCode.ActualHeight + barcodeH);
                }
                result.Append(readable.Item1);
            }

            result.Append("</Canvas>");
            readable = new Tuple<string, Uri, Stream, int>(result.ToString(), readable.Item2, readable.Item3, 0);
            return readable;
        }

        public static string GetBarcodeDescription(string content, string barcodeHeight, string x, string y)
        {
            var result = new StringBuilder();
            var bitMatrix = Encode(content);
            for (var i = 0; i < bitMatrix.Width; i++)
            {
                if (bitMatrix[i, 0])
                {
                    var currentLineEndX = GetEndX(bitMatrix, i);
                    result.AppendFormat(CultureInfo.InvariantCulture, "M {0},0 L {1},0 {1},{2} {0},{2} Z ", i, currentLineEndX, barcodeHeight);
                    i = currentLineEndX;
                }
            }
            var path = new Path
            {
                Data = Geometry.Parse(result.ToString()),
                Fill = new SolidColorBrush(Color.FromRgb(0, 0, 0))
            };
            if (!x.Equals("0"))
                Canvas.SetLeft(path, double.Parse(x, CultureInfo.InvariantCulture));
            if (!y.Equals("0"))
                Canvas.SetTop(path, double.Parse(y, CultureInfo.InvariantCulture));

            var textPath = XamlWriterForGraphicElement.Save(path);

            return textPath;
        }

        private static int GetEndX(BitMatrix matrix, int currentPosX)
        {
            int x;
            for (x = currentPosX + 1; x < matrix.Width; x++)
            {
                if (!matrix[x, 0])
                {
                    break;
                }
            }
            return x;
        }

        public Tuple<string, Uri, Stream, int> GetBarcodeForXps(BarcodeDescription barcode, Size pageSize)
        {
            if (barcode == null)
                return null;

            var result = new StringBuilder();

            if (barcode.Barcode != null && barcode.TextBlock != null)
                return GetCanvasFullBarcode(result, barcode, pageSize);

            if (barcode.Barcode != null && barcode.TextBlock == null)
                return GetCanvasForPathOnlyBarcode(result, barcode, pageSize);

            if (barcode.Barcode == null && barcode.TextBlock != null)
                return GetCanvasForTextOnlyBarcode(result, barcode, pageSize);

            throw new ArgumentNullException("barcode");
        }

        private Tuple<string, Uri, Stream, int> GetCanvasForTextOnlyBarcode(StringBuilder result, BarcodeDescription barcode, Size pageSize)
        {
            var offsetX = double.IsNaN(Canvas.GetLeft(barcode.TextBlock)) ? 0.0 : Canvas.GetLeft(barcode.TextBlock);
            var offsetY = double.IsNaN(Canvas.GetTop(barcode.TextBlock)) ? 0.0 : Canvas.GetTop(barcode.TextBlock);
            var readable = XpsWriter.GenerateGlyphsFromTextBlock(barcode.TextBlock);

            switch (barcode.HorizontalAlignment)
            {
                case HorizontalAlignment.Right:
                    offsetX = pageSize.Width + offsetX - barcode.TextBlock.DesiredSize.Width;
                    break;
                case HorizontalAlignment.Center:
                    offsetX = ((pageSize.Width - barcode.TextBlock.DesiredSize.Width) / 2) + offsetX;
                    break;
            }

            switch (barcode.VerticalAlignment)
            {
                case VerticalAlignment.Bottom:
                    offsetY = pageSize.Height + offsetY - barcode.TextBlock.DesiredSize.Height;
                    break;
                case VerticalAlignment.Center:
                    offsetY = ((pageSize.Height - barcode.TextBlock.DesiredSize.Height) / 2) + offsetY;
                    break;
            }

            result.AppendFormat(CultureInfo.InvariantCulture,
                "<Canvas Name=\"barcode\" RenderTransform=\"{2}, 0, 0, {2}, {0}, {1}\">",
                offsetX.ToString(CultureInfo.InvariantCulture),
                offsetY.ToString(CultureInfo.InvariantCulture), barcode.Scale);

            var textCanvas = readable.Item1;

            var textXml = XElement.Parse(textCanvas);
            textXml.Attribute("RenderTransform").Value = "1, 0, 0, 1, 0, 0";
            textCanvas = textXml.ToString();
            result.Append(textCanvas);
            result.Append("</Canvas>");

            readable = new Tuple<string, Uri, Stream, int>(result.ToString(), readable.Item2, readable.Item3, 0);
            return readable;
        }

        private Tuple<string, Uri, Stream, int> GetCanvasForPathOnlyBarcode(StringBuilder result, BarcodeDescription barcode, Size pageSize)
        {
            var offsetX = double.IsNaN(Canvas.GetLeft(barcode.Barcode)) ? 0.0 : Canvas.GetLeft(barcode.Barcode);
            var offsetY = double.IsNaN(Canvas.GetTop(barcode.Barcode)) ? 0.0 : Canvas.GetTop(barcode.Barcode);

            switch (barcode.HorizontalAlignment)
            {
                case HorizontalAlignment.Right:
                    offsetX = pageSize.Width + offsetX - barcode.Barcode.Data.Bounds.BottomRight.X;
                    break;
                case HorizontalAlignment.Center:
                    offsetX = ((pageSize.Width - barcode.Barcode.Data.Bounds.BottomRight.X) / 2) + offsetX;
                    break;
            }

            switch (barcode.VerticalAlignment)
            {
                case VerticalAlignment.Bottom:
                    offsetY = pageSize.Height + offsetY - barcode.Barcode.Data.Bounds.BottomRight.Y;
                    break;
                case VerticalAlignment.Center:
                    offsetY = ((pageSize.Height - barcode.Barcode.Data.Bounds.BottomRight.Y) / 2) + offsetY;
                    break;
            }

            result.AppendFormat(CultureInfo.InvariantCulture,
                "<Canvas Name=\"barcode\" RenderTransform=\"{2}, 0, 0, {2}, {0}, {1}\">",
                offsetX.ToString(CultureInfo.InvariantCulture),
                offsetY.ToString(CultureInfo.InvariantCulture), barcode.Scale);

            result.AppendFormat(CultureInfo.InvariantCulture, "<Path Data=\"{0}\" Fill=\"#000000\" />", barcode.Barcode.Data);
            result.AppendFormat(CultureInfo.InvariantCulture, "<Path Data=\"M 0,0 L {0},0 {0},{1} 0,{1} Z\" Opacity=\"0\"/>",
                barcode.Barcode.Data.Bounds.TopRight.X, barcode.Barcode.Data.Bounds.Bottom);

            result.Append("</Canvas>");

            return new Tuple<string, Uri, Stream, int>(result.ToString(), null, null, 0);
        }

        private Tuple<string, Uri, Stream, int> GetCanvasFullBarcode(StringBuilder result, BarcodeDescription barcode, Size pageSize)
        {
            var offsetX = double.IsNaN(Canvas.GetLeft(barcode.Barcode)) ? 0.0 : Canvas.GetLeft(barcode.Barcode);
            var offsetY = double.IsNaN(Canvas.GetTop(barcode.Barcode)) ? 0.0 : Canvas.GetTop(barcode.Barcode);

            switch (barcode.HorizontalAlignment)
            {
                case HorizontalAlignment.Right:
                    offsetX = pageSize.Width + offsetX - barcode.Barcode.Data.Bounds.BottomRight.X;
                    break;
                case HorizontalAlignment.Center:
                    offsetX = ((pageSize.Width - barcode.Barcode.Data.Bounds.BottomRight.X) / 2) + offsetX;
                    break;
            }

            switch (barcode.VerticalAlignment)
            {
                case VerticalAlignment.Bottom:
                    offsetY = pageSize.Height + offsetY - barcode.Barcode.Data.Bounds.BottomRight.Y;
                    break;
                case VerticalAlignment.Center:
                    offsetY = ((pageSize.Height - barcode.Barcode.Data.Bounds.BottomRight.Y) / 2) + offsetY;
                    break;
            }

            result.AppendFormat(CultureInfo.InvariantCulture,
                "<Canvas Name=\"barcode\" RenderTransform=\"{2}, 0, 0, {2}, {0}, {1}\">",
                offsetX.ToString(CultureInfo.InvariantCulture),
                offsetY.ToString(CultureInfo.InvariantCulture), barcode.Scale);

            result.AppendFormat(CultureInfo.InvariantCulture, "<Path Data=\"{0}\" Fill=\"#000000\" />", barcode.Barcode.Data);

            var readable = XpsWriter.GenerateGlyphsFromTextBlock(barcode.TextBlock);
            var width = barcode.Barcode.Data.Bounds.TopRight.X;
            var height = barcode.Barcode.Data.Bounds.Bottom + barcode.TextBlock.ActualHeight;
            result.AppendFormat(CultureInfo.InvariantCulture, "<Path Data=\"M 0,0 L {0},0 {0},{1} 0,{1} Z\" Opacity=\"0\"/>", width, height);

            var textCanvas = readable.Item1;

            var textWidth = barcode.TextBlock.ActualWidth;
            var textOffsetX = (barcode.Barcode.Data.Bounds.TopRight.X - textWidth) / 2;
            var textXml = XElement.Parse(textCanvas);
            textXml.Attribute("RenderTransform").Value = string.Format(CultureInfo.InvariantCulture,
                "1, 0, 0, 1, {0}, {1}",
                textOffsetX, barcode.Barcode.Data.Bounds.Bottom);
            textCanvas = textXml.ToString();
            result.Append(textCanvas);

            result.Append("</Canvas>");

            readable = new Tuple<string, Uri, Stream, int>(result.ToString(), readable.Item2, readable.Item3, 0);
            return readable;
        }

        Tuple<string, Uri, Stream, int> IBarCodeCreator.GetBarcodeInPathes(string content, string offsetX, string offsetY, string barcodeHeight, string scale, bool showText, bool textOnly)
        {
            return GetBarcodeInPathes(content, offsetX, offsetY, barcodeHeight, scale, showText, textOnly);
        }
    }
}
