/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.IO.Packaging;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Markup;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Xps.Packaging;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;
using Ascon.Pilot.Common;
using Ascon.Pilot.Common.Utils;
using Ascon.Pilot.DataClasses;
using Pilot.Xps.Domain.GraphicLayer;
using Pilot.Xps.Domain.IO;
using Pilot.Xps.Domain.Labels;
using Pilot.Xps.Domain.Print;
using Pilot.Xps.Domain.Tools;

namespace Pilot.Xps.Domain.Render
{
    class XpsWriter : IXpsWriter
    {
        public Stream InjectGraphicLayerElements(Stream sourceStream, List<GraphicLayerElement> graphicLayerElments)
        {
            if (graphicLayerElments == null || graphicLayerElments.Count == 0)
                return sourceStream;

            var destStream = IoUtils.CreateMemoryStream(sourceStream);

            using (var sourcePackage = Package.Open(sourceStream, FileMode.Open, FileAccess.Read))
            {
                using (var resultPackage = Package.Open(destStream, FileMode.OpenOrCreate, FileAccess.ReadWrite))
                {
                    var listoflists = new List<List<GraphicLayerElement>>();
                    foreach (var graphicLayerElement in graphicLayerElments.Where(l => l.ContentType == XpsConstants.XAML || l.ContentType == XpsConstants.BITMAP))
                    {
                        var t = listoflists.FirstOrDefault(list => list.Any(element => element.PageNumber == graphicLayerElement.PageNumber));
                        if (t != null)
                            t.Add(graphicLayerElement);
                        else
                        {
                            listoflists.Add(new List<GraphicLayerElement> { graphicLayerElement });
                        }
                    }
                    foreach (var graphicLayerElement in listoflists)
                    {
                        if (graphicLayerElement.Count == 0)
                            continue;
                        var sourcePagePart = GetSourcePagePart(sourcePackage, graphicLayerElement[0].PageNumber);
                        if (sourcePagePart == null)
                            return destStream;

                        var imagesFiles = new List<ImageInfo>(graphicLayerElement.Count);

                        using (var stream = sourcePagePart.GetStream())
                        {
                            var fixedPage = XElement.Load(stream);
                            var defaultNamespace = fixedPage.GetDefaultNamespace();

                            var glNode = fixedPage.Nodes().FirstOrDefault(n =>
                            {
                                var tmp = n as XElement;
                                return tmp != null &&
                                       tmp.Attributes("Name").Any(atr => atr.Value.Equals("graphicLayerElements"));
                            }) as XElement;
                            if (glNode == null)
                            {
                                var pathWithImages = CreatePathesWithImages(graphicLayerElement, imagesFiles, (double)fixedPage.Attribute("Height"), (double)fixedPage.Attribute("Width"));
                                fixedPage.Add(XpsTools.TryParseXml(pathWithImages));
                            }
                            else
                            {
                                var pathWithImages = GetListOfLayersForXps(graphicLayerElement, imagesFiles, (double)fixedPage.Attribute("Height"), (double)fixedPage.Attribute("Width"));
                                foreach (var pathWithImage in pathWithImages)
                                {
                                    glNode.Add(XpsTools.TryParseXml(pathWithImage));
                                }
                            }

                            foreach (var ce in fixedPage.DescendantsAndSelf())
                                ce.Name = defaultNamespace + ce.Name.LocalName;

                            var resultPagePart = resultPackage.GetPart(sourcePagePart.Uri);
                            using (var resultPagePartStream = resultPagePart.GetStream(FileMode.Create))
                            {
                                fixedPage.Save(resultPagePartStream);
                            }

                            foreach (var imagesFile in imagesFiles)
                            {

                                var imageSource = "/Resources/" + Path.GetFileName(imagesFile.Uri);
                                resultPagePart.CreateRelationship(new Uri(imageSource,
                                    UriKind.Relative), TargetMode.Internal, XpsConstants.REQUIRED_RESOURCE);
                            }
                        }

                        foreach (var imageInfo in imagesFiles)
                        {
                            var file = "/Resources/" + Path.GetFileName(imageInfo.Uri);

                            string contentType;
                            switch (imageInfo.XpsImageType)
                            {
                                case XpsImageType.PngImageType:
                                    contentType = "image/png";
                                    break;
                                case XpsImageType.JpegImageType:
                                    contentType = "image/jpeg";
                                    break;
                                default:
                                    throw new BadImageFormatException("Unknown image type");
                            }

                            using (var imageStream = XpsTools.GetStreamInPackageAllowingMultipleFiles(resultPackage, XpsConstants.REQUIRED_RESOURCE, file, contentType))
                            {
                                using (var stream = File.Open(imageInfo.Uri, FileMode.Open))
                                {
                                    stream.CopyTo(imageStream);
                                }
                                File.Delete(imageInfo.Uri);
                            }
                        }
                    }
                }
            }

            destStream.Position = 0;
            return destStream;
        }

        public List<Tuple<string, Uri, Stream, int>> GenerateGlyphsFromTextLabels(string textLabelsConfiguration, List<Size> pageSizes, Dictionary<string, object> attributesValues, List<string> attributeNames)
        {
            var textLabels = new TextLabelList(textLabelsConfiguration);
            if (!textLabels.Any())
                return null;
            var result = new List<Tuple<string, Uri, Stream, int>>();

            foreach (var textLabel in textLabels)
            {
                var textBlock = BarcodeLabelsDrawer.DrawTextBlock(textLabel);
                if (textBlock == null)
                    continue;

                var parsedPageRange = PagesPrintOrder.Parse(textLabel.PageRange, pageSizes.Count, 1);
                foreach (var page in parsedPageRange)
                {
                    var pageSize = pageSizes[page - 1];
                    var allValues = GetAttributesValues(attributeNames, attributesValues, textLabel.HideIfEmptyValue);
                    var textLines = textBlock.Inlines.ToList();
                    var formattedText = new StringBuilder();
                    foreach (var line in textLines)
                    {
                        if (line is Run)
                            formattedText.Append((line as Run).Text.Format(allValues));
                        if (line is LineBreak)
                            formattedText.AppendLine();
                    }

                    textBlock.Text = formattedText.ToString();

                    var entry = allValues.Where(e => e.Value != null && e.Value.ToString().StartsWith(NumeratorDescription.Deferred))
                        .Select(e => (KeyValuePair<string, object>?)e)
                        .FirstOrDefault();
                    if (entry != null && formattedText.ToString().Contains(NumeratorDescription.Deferred))
                        continue;

                    var generatedTextBlock = GenerateGlyphsFromTextLabel(textLabel, textBlock, pageSize.Width, pageSize.Height);
                    if (generatedTextBlock != null)
                        result.Add(new Tuple<string, Uri, Stream, int>(generatedTextBlock.Item1, generatedTextBlock.Item2, generatedTextBlock.Item3, page - 1));
                }
            }

            return !result.Any() ? null : result;
        }

        public Stream AddTextLabelsAndBarcode(Stream sourceStream, List<Tuple<string, Uri, Stream, int>> textLabels, Tuple<Tuple<string, Uri, Stream, int>, HorizontalAlignment, VerticalAlignment, double> barcode = null)
        {
            if (textLabels == null && barcode == null)
                return sourceStream;

            var destStream = IoUtils.CreateMemoryStream(sourceStream);
            using (var sourcePackage = Package.Open(sourceStream, FileMode.Open, FileAccess.Read))
            {
                using (var resultPackage = Package.Open(destStream, FileMode.OpenOrCreate, FileAccess.ReadWrite))
                {
                    var fixedDocInfo = new FixedDocInfo();
                    fixedDocInfo.FixedDocContents = new List<FixedDocContent>();
                    var xmlSerializer = new XmlSerializer(typeof(FixedDocInfo));

                    var fixedDocInfoExist = XpsTools.IsDataFileExist(resultPackage, XpsConstants.FIXED_DOC_INFO_DATA_FILE, XpsConstants.FIXED_DOC_INFO_DATA_FILE, XpsConstants.FIXED_DOC_INFO_CONTENT_TYPE);

                    if (fixedDocInfoExist)
                    {
                        using (var fixedDocInfoExistingStream = XpsTools.GetStreamInPackage(resultPackage, XpsConstants.FIXED_DOC_INFO_DATA_FILE, XpsConstants.FIXED_DOC_INFO_DATA_FILE, XpsConstants.FIXED_DOC_INFO_CONTENT_TYPE, false))
                        {
                            var doc = xmlSerializer.Deserialize(fixedDocInfoExistingStream) as FixedDocInfo;
                            if (doc != null)
                                fixedDocInfo = doc;
                        }
                    }
                    var fontResourceParts = new List<PackagePart>();

                    using (var fixedDocInfoStream = XpsTools.GetStreamInPackage(resultPackage, XpsConstants.FIXED_DOC_INFO_DATA_FILE, XpsConstants.FIXED_DOC_INFO_DATA_FILE, XpsConstants.FIXED_DOC_INFO_CONTENT_TYPE, true))
                    {
                        if (textLabels != null && textLabels.Any())
                        {
                            fixedDocInfo.FixedDocContents.Add(new FixedDocContent { Name = BarcodeLabelsConstants.TEXT_LABEL_ATTRIBUTE_NAME, Pages = new List<int>() { 0 } });

                            foreach (var textLabel in textLabels)
                            {
                                if (textLabel != null && textLabel.Item2 != null && textLabel.Item3 != null)
                                {
                                    fontResourceParts.Add(AddFontResourceToPackage(resultPackage, textLabel.Item2, textLabel.Item3));
                                }
                            }
                        }

                        if (barcode != null && barcode.Item1 != null && barcode.Item1.Item2 != null && barcode.Item1.Item3 != null)
                        {
                            fontResourceParts.Add(AddFontResourceToPackage(resultPackage, barcode.Item1.Item2, barcode.Item1.Item3));
                            fixedDocInfo.FixedDocContents.Add(new FixedDocContent { Name = BarcodeLabelsConstants.BAR_CODE_ATTRIBUTE_NAME, Pages = new List<int>() { 0 } });
                        }

                        using (var writer = new StreamWriter(fixedDocInfoStream))
                        {
                            xmlSerializer.Serialize(writer, fixedDocInfo);
                            writer.Flush();
                        }
                    }

                    var pages = sourcePackage.GetPages();

                    for (var i = 0; i < pages.Count; i++)
                    {
                        var sourcePagePart = pages[i];
                        if (sourcePagePart == null)
                            continue;

                        using (var stream = sourcePagePart.GetStream())
                        {
                            var fixedPage = XElement.Load(stream);
                            var defaultNamespace = fixedPage.GetDefaultNamespace();


                            // NOTE [Kolpakov A. 16.07.2015]: 
                            // workaround for "print/save as image" floating barcode
                            // if constant barcode was already injected
                            var nameAttrs = fixedPage.Descendants().Where(e => e.Attribute("Name") != null).ToList();

                            var injectedBarcode = nameAttrs.FirstOrDefault(e => e.Attribute("Name").Value.Equals(BarcodeLabelsConstants.BAR_CODE_ATTRIBUTE_NAME));
                            if (injectedBarcode != null)
                            {
                                injectedBarcode.Remove();
                                if (barcode == null)
                                    fixedPage.Add(injectedBarcode);
                            }

                            RotateBarcodeOnPage(ref barcode);

                            // NOTE [Kolpakov A. 08.07.2015]: 
                            // workaround for "print/save as image" floating text labels
                            // if constant labels was already injected
                            var injectedLabels = nameAttrs.FirstOrDefault(e => e.Attribute("Name").Value.Equals(BarcodeLabelsConstants.TEXT_LABEL_ATTRIBUTE_NAME));
                            var pageNumber = i;
                            if (textLabels != null && textLabels.Any())
                            {
                                var textLabelsOnPage = textLabels.Where(t => t.Item4 == pageNumber).ToList();
                                var labelsInCanvas = XpsTools.TryParseXml(GetLabelsInCanvas(textLabelsOnPage, BarcodeLabelsConstants.TEXT_LABEL_ATTRIBUTE_NAME));

                                if (labelsInCanvas != null)
                                {
                                    if (injectedLabels != null)
                                    {
                                        foreach (var label in injectedLabels.Nodes())
                                            labelsInCanvas.Add(label as XElement);
                                        injectedLabels.Remove();
                                    }
                                    fixedPage.Add(labelsInCanvas);
                                }
                            }

                            if (barcode != null && barcode.Item1 != null && barcode.Item1.Item4 == i)
                            {
                                var barcodeParsed = XpsTools.TryParseXml(barcode.Item1.Item1);
                                if (barcodeParsed != null)
                                    fixedPage.Add(barcodeParsed);
                            }

                            foreach (var ce in fixedPage.DescendantsAndSelf())
                                ce.Name = defaultNamespace + ce.Name.LocalName;

                            var resultPagePart = resultPackage.GetPart(sourcePagePart.Uri);
                            using (var resultPagePartStream = resultPagePart.GetStream(FileMode.Create))
                            {
                                fixedPage.Save(resultPagePartStream);
                            }

                            foreach (var fontResourcePart in fontResourceParts)
                            {
                                resultPagePart.CreateRelationship(fontResourcePart.Uri, TargetMode.Internal, XpsConstants.REQUIRED_RESOURCE);
                            }
                        }
                    }
                }
            }

            destStream.Position = 0;
            return destStream;
        }

        private string CreatePathesWithImages(List<GraphicLayerElement> graphicLayerElements, List<ImageInfo> imagesFiles, double pageHeight, double pageWidth)
        {
            var r = new StringBuilder();
            r.AppendFormat(CultureInfo.InvariantCulture, "<Canvas Name=\"{0}\" RenderTransform=\"1, 0, 0, 1, 0, 0\">", "graphicLayerElements");
            r.Append(GetListOfLayersForXps(graphicLayerElements, imagesFiles, pageHeight, pageWidth).Aggregate((i, j) => i + j));
            r.Append("</Canvas>");
            return r.ToString();
        }

        private List<string> GetListOfLayersForXps(List<GraphicLayerElement> graphicLayerElements, List<ImageInfo> imagesFiles, double pageHeight, double pageWidth)
        {
            var result = new List<string>();
            foreach (var element in graphicLayerElements)
            {
                var filename = Path.Combine(SpecialDirectoryProvider.GetTempPath(), Guid.NewGuid() + ".png");
                switch (element.ContentType)
                {
                    case XpsConstants.BITMAP:
                        {
                            var rastrElement = element.GraphicLayerElementContent as RasterGraphicLayerElementContent;
                            using (var file = File.Create(filename))
                                rastrElement.GetStream().CopyTo(file);
                            break;
                        }
                    case XpsConstants.XAML:
                        RenderTargetBitmap targetBitamp = null;
                        
                        var content = element.GraphicLayerElementContent.GetStream();
                        var fact = new GraphicElementContentFactory();
                        var contr = fact.MakeContent(XpsConstants.XAML, content);
                        contr.Element.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
                        contr.Element.Arrange(new Rect(contr.Element.DesiredSize));

                        var bounds = VisualTreeHelper.GetDescendantBounds(contr.Element);
                        if (bounds.IsEmpty) // На случай, если будет элемент без контента. Нельзя, чтобы был пустой Rect
                            bounds = new Rect(new Size(1, 1));

                        var dpiScale = 300.0 / 96;
                        var dpiX = 300.0;
                        var dpiY = 300.0;
                        targetBitamp = new RenderTargetBitmap((int)(bounds.Width * dpiScale), (int)(bounds.Height * dpiScale), dpiX, dpiY, PixelFormats.Pbgra32);
                        var drawingVisual = new DrawingVisual();
                        using (var ctx = drawingVisual.RenderOpen())
                        {
                            var visualBrush = new VisualBrush(contr.Element);
                            ctx.DrawRectangle(visualBrush, null, new Rect(new Point(), bounds.Size));
                        }
                        targetBitamp.Render(drawingVisual);
                        targetBitamp.Freeze();

                        var encoder = new PngBitmapEncoder();
                        encoder.Frames.Add(BitmapFrame.Create(targetBitamp));
                        using (Stream stream = File.Create(filename))
                            encoder.Save(stream);
                        break;
                }
                var imageSource = "/Resources/" + Path.GetFileName(filename);
                var imageInfo = ImageHelper.GetImageInfo(filename);
                imagesFiles.Add(imageInfo);

                var imageInfoW = imageInfo.Width * 96.0 / imageInfo.HorizontalResolution;
                var imageInfoH = imageInfo.Height * 96.0 / imageInfo.VerticalResolution;

                var imageHeight = imageInfoH * element.Scale.X;
                var imageWidth = imageInfoW * element.Scale.Y;

                var offsetX = element.OffsetX;
                var offsetY = element.OffsetY;

                switch (element.HorizontalAlignment)
                {
                    case HorizontalAlignment.Right:
                        offsetX = pageWidth - offsetX - imageWidth;
                        break;
                    case HorizontalAlignment.Center:
                        offsetX = (pageWidth / 2) + offsetX;
                        break;
                }

                switch (element.VerticalAlignment)
                {
                    case VerticalAlignment.Bottom:
                        offsetY = pageHeight - offsetY - imageHeight;
                        break;
                    case VerticalAlignment.Center:
                        offsetY = (pageHeight / 2) + offsetY;
                        break;
                }

                var m = new Matrix(element.Scale.X, 0, 0, element.Scale.Y, offsetX, offsetY);
                m.RotateAt(element.Angle, offsetX, offsetY);
                var r = new StringBuilder();
                r.AppendFormat(CultureInfo.InvariantCulture,
                    "<Canvas RenderTransform=\"{0}, {1}, {2}, {3}, {4}, {5}\">", m.M11, m.M12,
                    m.M21, m.M22,
                    m.OffsetX, m.OffsetY, "graphicLayerElement" + element.ElementId.ToString("N"));

                var path = string.Format(new CultureInfo("en-US"),
                    @"<Path Data=""F0 M 0, 0 L {0}, 0 {0}, {1} 0, {1}Z"">
                                <Path.Fill>
                                    <ImageBrush ViewportUnits=""Absolute"" TileMode=""None"" ViewboxUnits=""Absolute"" Viewbox=""0,0,{0},{1}"" Viewport=""0,0,{0},{1}"" ImageSource=""{2}"" />
                                </Path.Fill>
                                </Path>", imageInfoW, imageInfoH, imageSource);
                r.Append(path);
                r.Append("</Canvas>");
                result.Add(r.ToString());
            }
            return result;
        }

        private static PackagePart GetSourcePagePart(Package sourcePackage, int pageNumber)
        {
            var pages = sourcePackage.GetPages();
            if (pageNumber > pages.Count)
                throw new Exception($"Page {pageNumber + 1} not found");

            var fixedDocParts = sourcePackage.GetParts().Where(sourcePart => sourcePart.ContentType == XpsConstants.FIXED_DOC_CONTENT_TYPE).ToList();
            for (int i = 0; i < fixedDocParts.Count; i++)
            {
                var fixedDocPart = fixedDocParts[i];
                if (fixedDocPart != null)
                {
                    using (var stream = fixedDocPart.GetStream())
                    {
                        var fixedDoc = XElement.Load(stream);
                        var ns = fixedDoc.GetDefaultNamespace();
                        var pageContents = fixedDoc.Descendants(ns + "PageContent").ToList();
                        if (pageNumber >= pageContents.Count)
                        {
                            pageNumber = pageNumber - pageContents.Count;
                            continue;
                        }

                        var pageToRotate = pageContents.ElementAtOrDefault(pageNumber);
                        if (pageToRotate == null)
                            throw new Exception($"Page {pageNumber + 1} not found");

                        var pageToRotateName = pageToRotate.Attribute("Source").Value;

                        var foundPageParts = pages.Where(p => p.Uri.OriginalString.EndsWith(pageToRotateName)).ToList();

                        var sourcePagePart = foundPageParts[i];

                        if (sourcePagePart == null)
                            throw new Exception($"Page {pageNumber + 1} not found");

                        return sourcePagePart;
                    }
                }
            }

            return null;
        }

        public static Tuple<string, Uri, Stream, int> GenerateGlyphsFromTextBlock(TextBlock textBlock)
        {
            return GenerateGlyphsFromTextLabel(BarcodeLabelsDrawer.GeTextLabelFromTextBlock(textBlock), textBlock, 0, 0);
        }

        public static Tuple<string, Uri, Stream, int> GenerateGlyphsFromTextLabel(TextLabel textLabel, TextBlock textBlock, double pageWidth, double pageHeight)
        {
            Canvas.SetTop(textBlock, 0);
            Canvas.SetLeft(textBlock, 0);
            Canvas.SetRight(textBlock, double.NaN);
            Canvas.SetBottom(textBlock, double.NaN);

            var text = new StringBuilder();
            var xmlWriter = XmlWriter.Create(text, new XmlWriterSettings { OmitXmlDeclaration = true, ConformanceLevel = ConformanceLevel.Fragment });

            GenerateAlignedAndRotatedCanvas(xmlWriter, textLabel, textBlock, pageWidth, pageHeight);

            using (var resultPackage = WriteToEmptyXpsPackage(new[] { textBlock }, pageWidth, pageHeight))
            {
                var pagePart = resultPackage.GetParts().FirstOrDefault(p => p.ContentType == XpsConstants.FIXED_PAGE_CONTENT_TYPE);
                var fontPart = resultPackage.GetParts().FirstOrDefault(p => p.ContentType == XpsConstants.OBFUSCATED_OPENTYPE);
                if (fontPart != null)
                {
                    var fontStream = IoUtils.CreateMemoryStream(fontPart.GetStream());

                    using (var reader = XmlReader.Create(pagePart.GetStream()))
                    {
                        while (reader.Read())
                        {
                            while (reader.Name.Equals("Glyphs"))
                            {
                                xmlWriter.WriteRaw(reader.ReadOuterXml());
                            }
                        }
                        xmlWriter.WriteRaw("</Canvas>");
                        xmlWriter.Close();
                        return new Tuple<string, Uri, Stream, int>(text.ToString(), fontPart.Uri, fontStream, 0);
                    }
                }
                return null;
            }
        }

        private static void GenerateAlignedAndRotatedCanvas(XmlWriter xmlWriter, TextLabel textLabel, TextBlock textBlock, double pageWidth, double pageHeight)
        {
            var offsetX = 0d;
            var offsetY = 0d;
            var textSize = BarcodeLabelsDrawer.MeasureTextblock(textBlock, textBlock.Text);

            var tbWidth = double.IsNaN(textBlock.Width) ? textSize.Width : textBlock.Width;
            var tbHeight = double.IsNaN(textBlock.Height) ? textSize.Height : textBlock.Height;

            var tbSize = new Size(tbWidth, tbHeight);

            switch (textLabel.HorizontalAlignment)
            {
                case HorizontalAlignment.Left:
                    CultureInvariantDoublesParseHelper.TryParseDouble(textLabel.LeftValue, out var left);
                    offsetX = left;
                    Canvas.SetLeft(textBlock, 0);
                    break;
                case HorizontalAlignment.Center:
                    CultureInvariantDoublesParseHelper.TryParseDouble(textLabel.LeftValue, out var center);
                    offsetX = ((pageWidth - tbWidth) / 2) + center;
                    Canvas.SetLeft(textBlock, 0);
                    break;
                case HorizontalAlignment.Right:
                    CultureInvariantDoublesParseHelper.TryParseDouble(textLabel.RightValue, out var right);
                    Canvas.SetRight(textBlock, right);
                    offsetX = pageWidth - right - tbWidth;
                    break;
                case HorizontalAlignment.Stretch:
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }

            switch (textLabel.VerticalAlignment)
            {
                case VerticalAlignment.Top:
                    CultureInvariantDoublesParseHelper.TryParseDouble(textLabel.TopValue, out var top);
                    offsetY = top;
                    Canvas.SetTop(textBlock, 0);
                    break;
                case VerticalAlignment.Center:
                    CultureInvariantDoublesParseHelper.TryParseDouble(textLabel.TopValue, out var center);
                    offsetY = ((pageHeight - tbHeight) / 2) + center;
                    Canvas.SetTop(textBlock, 0);
                    break;
                case VerticalAlignment.Bottom:
                    CultureInvariantDoublesParseHelper.TryParseDouble(textLabel.BottomValue, out var bottom);
                    Canvas.SetBottom(textBlock, bottom);
                    offsetY = pageHeight - bottom - tbHeight;
                    break;
                case VerticalAlignment.Stretch:
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }

            if (textLabel.KeepWithinPageBounds)
            {
                var point = BarcodeLabelsDrawer.CorrectNegativePoint(offsetX, offsetY);
                offsetX = point.X;
                offsetY = point.Y;
                offsetX = CheckCoordinatesForAdorners.CheckX(offsetX, offsetY, tbSize, new Size(pageWidth, pageHeight), textLabel.Angle);
                offsetY = CheckCoordinatesForAdorners.CheckY(offsetX, offsetY, tbSize, new Size(pageWidth, pageHeight), textLabel.Angle);
            }

            var r = new double[6] { 1, 0, 0, 1, offsetX, offsetY };
            var m = new Matrix(r[0], r[1], r[2], r[3], r[4], r[5]);
            m.RotateAt(textLabel.Angle, r[4], r[5]);
            xmlWriter.WriteRaw(string.Format(CultureInfo.InvariantCulture, "<Canvas RenderTransform=\"{0}, {1}, {2}, {3}, {4}, {5}\">", m.M11, m.M12, m.M21, m.M22, m.OffsetX, m.OffsetY));
        }

        public Size GetFirstPageSizeFromXps(Stream docStream)
        {
            return GetFirstPageSizeS(docStream);
        }

        public static Size GetFirstPageSizeS(Stream docStream)
        {
            if (docStream == Stream.Null)
                return Size.Empty;

            var pageSize = new Size(0, 0);

            using (var sourcePackage = Package.Open(docStream))
            {
                var sourcePagePart = GetSourcePagePart(sourcePackage, 0);
                using (var reader = XmlReader.Create(sourcePagePart.GetStream()))
                {
                    while (reader.Read())
                    {
                        if (!reader.Name.Equals("FixedPage"))
                            continue;

                        var widthAttr = reader.GetAttribute("Width");
                        if (widthAttr != null && reader.NodeType == XmlNodeType.Element)
                            pageSize.Width = double.Parse(widthAttr, CultureInfo.InvariantCulture);

                        var heightAttr = reader.GetAttribute("Height");
                        if (heightAttr != null && reader.NodeType == XmlNodeType.Element)
                            pageSize.Height = double.Parse(heightAttr, CultureInfo.InvariantCulture);

                        break;
                    }
                }
            }
            return pageSize;
        }

        public static Size GetPageSize(Stream docStream, int pageNumber)
        {
            if (docStream == Stream.Null)
                return Size.Empty;

            var pageSize = new Size(0, 0);

            using (var sourcePackage = Package.Open(docStream))
            {
                var sourcePagePart = GetSourcePagePart(sourcePackage, pageNumber);
                using (var reader = XmlReader.Create(sourcePagePart.GetStream()))
                {
                    while (reader.Read())
                    {
                        if (!reader.Name.Equals("FixedPage"))
                            continue;

                        var widthAttr = reader.GetAttribute("Width");
                        if (widthAttr != null && reader.NodeType == XmlNodeType.Element)
                            pageSize.Width = double.Parse(widthAttr, CultureInfo.InvariantCulture);

                        var heightAttr = reader.GetAttribute("Height");
                        if (heightAttr != null && reader.NodeType == XmlNodeType.Element)
                            pageSize.Height = double.Parse(heightAttr, CultureInfo.InvariantCulture);

                        break;
                    }
                }
            }
            return pageSize;
        }

        public List<Size> GetPagesSizes(Stream docStream)
        {
            return GetPageSizes(docStream);
        }

        public static List<Size> GetPageSizes(Stream docStream)
        {
            if (docStream == Stream.Null)
                return new List<Size>();

            var result = new List<Size>();

            using (var sourcePackage = Package.Open(docStream))
            {
                var pages = sourcePackage.GetPages();
                var fixedDocParts = sourcePackage.GetParts().Where(sourcePart => sourcePart.ContentType == XpsConstants.FIXED_DOC_CONTENT_TYPE).ToList();

                foreach (var fixedDocPart in fixedDocParts)
                {
                    if (fixedDocPart == null)
                        continue;

                    using (var fixedDocStream = XmlReader.Create(fixedDocPart.GetStream()))
                    {
                        var fixedDoc = XElement.Load(fixedDocStream);
                        var ns = fixedDoc.GetDefaultNamespace();
                        var pageContents = fixedDoc.Descendants(ns + "PageContent").ToList();
                        foreach (var pageContent in pageContents)
                        {
                            var pageSourceName = pageContent.Attribute("Source").Value;
                            var fixedPage = pages.Where(p => p.Uri.OriginalString.EndsWith(pageSourceName)).ToList().FirstOrDefault();
                            if (fixedPage == null)
                                continue;

                            var pageSize = new Size(0, 0);

                            using (var reader = XmlReader.Create(fixedPage.GetStream()))
                            {
                                while (reader.Read())
                                {
                                    if (!reader.Name.Equals("FixedPage"))
                                        continue;

                                    var widthAttr = reader.GetAttribute("Width");
                                    if (widthAttr != null && reader.NodeType == XmlNodeType.Element)
                                        pageSize.Width = double.Parse(widthAttr, CultureInfo.InvariantCulture);

                                    var heightAttr = reader.GetAttribute("Height");
                                    if (heightAttr != null && reader.NodeType == XmlNodeType.Element)
                                        pageSize.Height = double.Parse(heightAttr, CultureInfo.InvariantCulture);

                                    break;
                                }
                            }

                            result.Add(pageSize);
                        }
                    }
                }
            }
            return result;
        }

        private static PackagePart AddFontResourceToPackage(Package resultPackage, Uri fontUri, Stream fontStream)
        {
            var fontResourcePart = resultPackage.CreatePart(PackUriHelper.CreatePartUri(fontUri), XpsConstants.OBFUSCATED_OPENTYPE);
            if (fontResourcePart == null)
                return null;
            using (var destinationStream = fontResourcePart.GetStream())
            using (fontStream)
            {
                fontStream.CopyTo(destinationStream);
            }
            return fontResourcePart;
        }

        private static string GetLabelsInCanvas(List<Tuple<string, Uri, Stream, int>> textLabels, string textBlockCanvasName)
        {
            if (textLabels == null || textLabels.Count == 0)
                return "";
            var r = new StringBuilder();
            r.AppendFormat(CultureInfo.InvariantCulture, "<Canvas Name=\"{0}\" RenderTransform=\"1, 0, 0, 1, 0, 0\">", textBlockCanvasName);
            foreach (var textLabel in textLabels)
            {
                r.Append(textLabel.Item1);
            }
            r.Append("</Canvas>");

            return r.ToString();
        }

        private static void RotateBarcodeOnPage(ref Tuple<Tuple<string, Uri, Stream, int>, HorizontalAlignment, VerticalAlignment, double> barcode)
        {
            if (barcode == null || barcode.Item1 == null)
                return;

            var barcodeCanvas = XElement.Parse(barcode.Item1.Item1);
            var renderTransform = barcodeCanvas.Attribute("RenderTransform");
            var transform = renderTransform.Value.Split(',');
            var r = transform.Select(t => double.Parse(t, CultureInfo.InvariantCulture)).ToArray();
            var m = new Matrix(r[0], r[1], r[2], r[3], r[4], r[5]);

            m.RotateAt(barcode.Item4, r[4], r[5]);
            renderTransform.SetValue(string.Format(CultureInfo.InvariantCulture, "{0},{1},{2},{3},{4},{5}", m.M11, m.M12, m.M21, m.M22, m.OffsetX, m.OffsetY));

            barcode = new Tuple<Tuple<string, Uri, Stream, int>, HorizontalAlignment, VerticalAlignment, double>(new Tuple<string, Uri, Stream, int>(barcodeCanvas.ToString(), barcode.Item1.Item2, barcode.Item1.Item3, 0), barcode.Item2, barcode.Item3, barcode.Item4);
        }

        private static Package WriteToEmptyXpsPackage(IEnumerable<TextBlock> textBlocks, double pageWidth, double pageHeight)
        {
            using (var sourcePackage = Package.Open(GetEmptyXps(pageWidth, pageHeight), FileMode.Open, FileAccess.ReadWrite))
            {
                return sourcePackage.WithRegistered(CompressionOption.Normal, sourceDocument =>
                {
                    var sourceReader = sourceDocument.FixedDocumentSequenceReader;

                    var resultPackage = Package.Open(new MemoryStream(), FileMode.Create, FileAccess.ReadWrite);
                    return resultPackage.WithRegistered(CompressionOption.Normal, resultDoc =>
                    {
                        var writer = XpsDocument.CreateXpsDocumentWriter(resultDoc);
                        var sourceFixedDoc = sourceReader.FixedDocuments.FirstOrDefault();
                        var sourceFirstPage = sourceFixedDoc.FixedPages.FirstOrDefault();

                        var pageContent = new PageContent();
                        ((IUriContext)pageContent).BaseUri = PackUriHelper.Create(sourceDocument.Uri, sourceFixedDoc.Uri);
                        pageContent.Source = PackUriHelper.GetRelativeUri(sourceFixedDoc.Uri, sourceFirstPage.Uri);

                        var fp = pageContent.GetPageRoot(false);
                        var width = fp.Width;
                        var height = fp.Height;
                        var sz = new Size(width, height);
                        fp.Measure(sz);
                        fp.Arrange(new Rect(new Point(), sz));
                        fp.UpdateLayout();
                        var canvas = new Canvas();
                        foreach (var textBlock in textBlocks)
                        {
                            canvas.Children.Add(textBlock);
                            textBlock.Measure(sz);
                        }
                        canvas.Measure(sz);
                        var rect = new Rect(new Point(), sz);
                        canvas.Arrange(rect);

                        canvas.UpdateLayout();

                        writer.Write(canvas);
                        canvas.Children.Clear();
                        return resultPackage;
                    });
                });
            }
        }

        private static MemoryStream GetEmptyXps(double pageWidth, double pageHeight)
        {
            var tempStream = new MemoryStream();
            using (var resultPackage = Package.Open(tempStream, FileMode.Create, FileAccess.ReadWrite))
            {
                resultPackage.WithRegistered(resultDoc =>
                {
                    // Add Document Sequence as root
                    IXpsFixedDocumentSequenceWriter documentSequenceWriter = resultDoc.AddFixedDocumentSequence();
                    // Add Document to Document Sequence
                    IXpsFixedDocumentWriter fixedDocumentWriter = documentSequenceWriter.AddFixedDocument();
                    IXpsFixedPageWriter fixedPageWriter = fixedDocumentWriter.AddFixedPage();
                    // формируем страницу
                    var pageContents = string.Format(new CultureInfo("en-US"), // чтобы десятичный разделитель был точкой
                        @"<FixedPage xmlns=""{0}"" xmlns:x=""{1}"" xml:lang=""und"" Width=""{2}"" Height=""{3}"">
                            <FixedPage.Resources>    
                            </FixedPage.Resources> 
                        </FixedPage>", XpsConstants.XPS_XMLN, XpsConstants.RESOURCEDICTIONARY_KEY, pageWidth, pageHeight);

                    XmlWriter xmlWriter = fixedPageWriter.XmlWriter;

                    xmlWriter.WriteRaw(pageContents);
                    fixedPageWriter.Commit();
                    fixedDocumentWriter.Commit();
                    documentSequenceWriter.Commit();
                });
            }
            tempStream.Position = 0;
            return tempStream;
        }

        public static string GetConstantTextLabel(string xmlString)
        {
            return GetFilteredTextLabel(xmlString, true);
        }

        public static string GetFloatingTextLabel(string xmlString)
        {
            return GetFilteredTextLabel(xmlString, false);
        }

        private static string GetFilteredTextLabel(string xmlString, bool removeFloating)
        {
            if (string.IsNullOrEmpty(xmlString))
                return xmlString;

            var xElement = XElement.Parse(xmlString);
            xElement.Elements().Where(Predicate(removeFloating)).Remove();

            return !xElement.Elements().Any() ? null : xElement.ToString();
        }

        private static string GetFilteredBarcode(string xmlString, bool isFloating)
        {
            if (string.IsNullOrEmpty(xmlString))
                return xmlString;

            var xElement = XElement.Parse(xmlString);
            return Predicate(isFloating).Invoke(xElement) ? xElement.ToString() : null;
        }

        public static string GetConstantBarcode(string xmlString)
        {
            return GetFilteredBarcode(xmlString, false);
        }

        public static string GetFloatingBarcode(string xmlString)
        {
            return GetFilteredBarcode(xmlString, true);
        }

        public static Dictionary<string, object> GetAttributesValues(List<string> attributesNames, Dictionary<string, object> attributesValues, bool canBeEmpty)
        {
            var allValues = new Dictionary<string, object>();
            if (attributesNames == null)
                return allValues;

            foreach (var attribute in attributesNames)
            {
                object attributeValue = null;
                if (attributesValues != null)
                    attributesValues.TryGetValue(attribute, out attributeValue);
                if (attributeValue == null)
                {
                    if (canBeEmpty)
                        attributeValue = string.Empty;
                    else
                        continue;
                }

                allValues[attribute] = attributeValue;
            }
            return allValues;
        }

        private static Func<XElement, bool> Predicate(bool isFloating)
        {
            return e => ((bool?)e.Attribute(BarcodeLabelsConstants.FLOATING_ATTRIBUTE)).GetValueOrDefault() == isFloating;
        }
    }
}
