/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Security.Cryptography.X509Certificates;

namespace Pilot.Xps.Domain.Signatures
{
    public class SignatureDisplayParams
    {
        private string _signerName;

        public bool IsSigned { get; set; }
        public bool IsValid { get; set; }

        public bool IsChecking { get; set; }

        public bool IsCertificateValid { get; set; }

        public bool CanUserSign { get; set; }

        public string SignerName
        {
            get => string.IsNullOrEmpty(_signerName) ? "Unknown position" : _signerName;
            set
            {
                _signerName = value;
                RequestedSigner = _signerName;
            }
        }

        public Guid? SignerId { get; set; }
        public string RequestedSigner { get; set; }
        public Guid? SignId { get; set; }
        public X509Certificate Certificate { get; set; }

        public bool IsChecked { get; set; }

        public string SignDate { get; set; }

        //private int _annotationsCount;

        public string Role { get; set; }

        public string NoSignText
        {
            get
            {
                var result = "No signature";
                return string.IsNullOrEmpty(Role) && !IsSigned ? result : $"{result} \"{Role}\"";
            }
        }

        public string SignedText
        {
            get
            {
                var result = "Signed";
                return string.IsNullOrEmpty(Role) && IsSigned ? result : string.Empty;
            }
        }

        public bool IsAdditional { get; set; }

        public override bool Equals(object obj)
        {
            var tmp = obj as SignatureDisplayParams;
            if (tmp == null)
                return false;
            var certificatesEquality = Certificate != null && tmp.Certificate != null && Certificate.Equals(tmp.Certificate) || (Certificate == null && tmp.Certificate == null);
            var dateEquality = SignDate != null && tmp.SignDate != null && SignDate.Equals(tmp.SignDate) || (SignDate == null && tmp.SignDate == null);
            var idEquality = SignId != null && tmp.SignId != null && SignId.Equals(tmp.SignId) || (SignId == null && tmp.SignId == null);
            var nameEquality = SignerName != null && tmp.SignerName != null && SignerName.Equals(tmp.SignerName) || (SignerName == null && tmp.SignerName == null);
            return certificatesEquality && dateEquality && idEquality && nameEquality;
        }

        public override int GetHashCode()
        {
            var certificateHashCode = Certificate != null ? Certificate.GetHashCode() : 0;
            var signDateHashCode = SignDate != null ? SignDate.GetHashCode() : 0;
            var signerIdHashCode = SignId != null ? SignId.GetHashCode() : 0;
            var signerNameHashCode = SignerName != null ? SignerName.GetHashCode() : 0;
            return certificateHashCode ^ signerIdHashCode ^ signDateHashCode ^ signerNameHashCode;
        }

        public void Assign(SignatureDisplayParams displayParams)
        {
            IsSigned = displayParams.IsSigned;
            IsValid = displayParams.IsValid;
            CanUserSign = displayParams.CanUserSign;
            SignerName = displayParams.SignerName;
            RequestedSigner = displayParams.RequestedSigner;
            SignerId = displayParams.SignerId;
            SignId = displayParams.SignId;
            Certificate = displayParams.Certificate;
            SignDate = displayParams.SignDate;
            IsChecked = false;
            IsAdditional = displayParams.IsAdditional;
        }
    }
}
