/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.IO.Packaging;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using System.Windows.Xps.Packaging;
using Ascon.Pilot.DataClasses;
using Pilot.Xps.Entities;

namespace Pilot.Xps.Domain.Signatures
{
    static class XpsSignatureHelper
    {
        public static void AddDefinitions(IList<XpsSignatureDefinition> signatureDefinitions, Stream xpsStream)
        {
            if (signatureDefinitions.Count == 0)
                return;

            using (var package = Package.Open(xpsStream, FileMode.Open, FileAccess.ReadWrite))
            {
                using (var xpsDoc = new XpsDocument(package))
                {
                    xpsDoc.AddDefinitions(signatureDefinitions.ToArray());
                }
            }
        }

        public static SignResult MergeDigitalSignatures(IList<DigitalSignature> digitalSignatures, Stream xpsStream)
        {
            if (digitalSignatures.Count == 0)
                return SignResult.SignedSuccessfully;
            using (var package = Package.Open(xpsStream, FileMode.Open, FileAccess.ReadWrite))
                return XpsSignatureUtils.SignPackage(package, digitalSignatures);
        }

        public static Collection<SignatureDisplayParams> SignRequests(INObject obj, IList<SignatureDisplayParams> signatureDisplayParams, Stream xpsStream, X509Certificate2 cert, IPilotSignatureProxy proxy)
        {
            var signedDisplayParams = new Collection<SignatureDisplayParams>();
            using (var package = Package.Open(xpsStream, FileMode.Open, FileAccess.ReadWrite))
            {
                using (var xpsDocument = new XpsDocument(package))
                {
                    var packageSignatureManager = new PackageDigitalSignatureManager(package);
                    packageSignatureManager.RemoveAllSignatures();
                    foreach (var signable in signatureDisplayParams)
                    {
                        var xpsSignature = xpsDocument.SignDigitally(cert, true, XpsDigSigPartAlteringRestrictions.None, signable.SignId.GetValueOrDefault());
                        var stubSignatureDefinition = new XpsSignatureDefinition { SpotId = Guid.NewGuid(), RequestedSigner = signable.SignerName };
                        var certificateName = new X509Certificate2(xpsSignature.SignerCertificate).GetNameInfo(X509NameType.DnsName, false);
                        signedDisplayParams.Add(new SignatureDisplayParams
                        {
                            SignId = signable.SignId,
                            Certificate = xpsSignature.SignerCertificate,
                            IsSigned = true,
                            IsChecking = true,
                            CanUserSign = false,
                            IsValid = xpsSignature.Verify() == VerifyResult.Success,
                            RequestedSigner = signable.SignerName,
                            SignerId = signable.SignerId,
                            SignDate = xpsSignature.SigningTime.ToLocalTime().ToString("g"),
                            SignerName = proxy.GetPersonTitleForSignature(obj, stubSignatureDefinition, certificateName, "")
                        });
                    }
                }
            }
            return signedDisplayParams;
        }

    }
}
