/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.IO.Packaging;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using System.Windows.Xps.Packaging;
using Ascon.Pilot.Common;
using Ascon.Pilot.DataClasses;

namespace Pilot.Xps.Domain.Signatures
{
    internal interface IXpsSignatureSettings
    {
        Collection<SignatureDisplayParams> GetSignatureDisplayParams(INObject document, Stream xpsStream);
        Collection<SignatureDisplayParams> GetSignatureDisplayParams(IList<INFile> files, Stream xpsStream);

        Collection<SignatureDisplayParams> SignAs(INObject document, IList<SignatureDisplayParams> signatureDisplayParams, DocumentContext context, X509Certificate2 cert, bool actualizeSignatures);
    }

    class XpsSignatureSettings : IXpsSignatureSettings
    {
        private readonly IPilotSignatureProxy _proxy;

        public XpsSignatureSettings(IPilotSignatureProxy proxy)
        {
            _proxy = proxy;
        }

        public Collection<SignatureDisplayParams> GetSignatureDisplayParams(INObject document, Stream xpsStream)
        {
            try
            {
                var signatureInfoList = new Collection<SignatureDisplayParams>();
                using (var p = Package.Open(xpsStream, FileMode.Open, FileAccess.Read))
                {
                    using (var xpsDoc = new XpsDocument(p))
                    {
                        var definitions = xpsDoc.GetSignatureDefinitions();
                        var packageSignatureManager = new PackageDigitalSignatureManager(p);
                        var signatures = packageSignatureManager.Signatures.Select(packageDigitalSignature => new XpsDigitalSignature(packageDigitalSignature, xpsDoc)).ToList();
                        var certificateChecker = new SignatureAndCertificateChecker();
                        
                        foreach (var sd in definitions)
                        {
                            var signature = signatures.FirstOrDefault(sign => sign.Id == sd.SpotId);
                            if (signature != null)
                            {
                                // Если запрос подписан:
                                signatures.Remove(signature);
                                var certificate2 = new X509Certificate2(signature.SignerCertificate);
                                var position = XpsSignatureUtils.GetPositionFromCertificate(certificate2.Subject);
                                signatureInfoList.Add(new SignatureDisplayParams
                                {
                                    Certificate = certificate2,
                                    SignerName = _proxy.GetPersonTitleForSignature(document, sd, certificate2.GetNameInfo(X509NameType.DnsName, false), position),
                                    RequestedSigner = sd.RequestedSigner,
                                    SignerId = _proxy.GetSignerId(document, sd),
                                    CanUserSign = false,
                                    IsSigned = true,
                                    IsChecking = true,
                                    IsValid = certificateChecker.CheckSignature(signature),
                                    SignId = sd.SpotId,
                                    Role = sd.Intent,
                                    SignDate = signature.SigningTime.ToLocalTime().ToString("g"),
                                    IsCertificateValid = certificateChecker.CheckCertificate(certificate2)
                                });
                            }
                            else
                            {
                                signatureInfoList.Add(new SignatureDisplayParams
                                {
                                    Certificate = null,
                                    SignerName = _proxy.GetPersonTitleForDefinition(document, sd),
                                    RequestedSigner = sd.RequestedSigner,
                                    SignerId = _proxy.GetSignerId(document, sd),
                                    CanUserSign = _proxy.CanSign(document, sd),
                                    IsSigned = false,
                                    IsValid = false,
                                    IsChecking = false,
                                    SignId = sd.SpotId,
                                    Role = sd.Intent
                                });
                            }
                        }

                        // Список подписей, которые содержатся в документе БЕЗ запросов:
                        signatures.ForEach(s =>
                        {
                            var certificate2 = new X509Certificate2(s.SignerCertificate);
                            var position = XpsSignatureUtils.GetPositionFromCertificate(certificate2.Subject);
                            signatureInfoList.Add(new SignatureDisplayParams
                            {
                                Certificate = certificate2,
                                SignerName = _proxy.GetPersonTitleForSignature(document, null, certificate2.GetNameInfo(X509NameType.DnsName, false), position),
                                CanUserSign = false,
                                IsSigned = true,
                                IsChecking = true,
                                SignDate = s.SigningTime.ToLocalTime().ToString("g"),
                                IsValid = certificateChecker.CheckSignature(s),
                                IsCertificateValid = certificateChecker.CheckCertificate(certificate2),
                                SignId = s.Id
                            });
                        });
                    }
                }
                return signatureInfoList;
            }
            catch (Exception ex)
            {
                throw new SignatureException("Error reading signatures", ex);
            }
        }

        public Collection<SignatureDisplayParams> GetSignatureDisplayParams(IList<INFile> files, Stream xpsStream)
        {
            try
            {
                var signatureInfoList = new Collection<SignatureDisplayParams>();
                using (var p = Package.Open(xpsStream, FileMode.Open, FileAccess.Read))
                {
                    using (var xpsDoc = new XpsDocument(p))
                    {
                        var definitions = xpsDoc.GetSignatureDefinitions();
                        var packageSignatureManager = new PackageDigitalSignatureManager(p);
                        var signatures = packageSignatureManager.Signatures.Select(packageDigitalSignature => new XpsDigitalSignature(packageDigitalSignature, xpsDoc)).ToList();
                        var certificateChecker = new SignatureAndCertificateChecker();

                        foreach (var sd in definitions)
                        {
                            var signature = signatures.FirstOrDefault(sign => sign.Id == sd.SpotId);
                            if (signature != null)
                            {
                                // Если запрос подписан:
                                signatures.Remove(signature);
                                var certificate2 = new X509Certificate2(signature.SignerCertificate);
                                var position = XpsSignatureUtils.GetPositionFromCertificate(certificate2.Subject);
                                signatureInfoList.Add(new SignatureDisplayParams
                                {
                                    Certificate = certificate2,
                                    SignerName = _proxy.GetPersonTitleForSignature(files, sd, certificate2.GetNameInfo(X509NameType.DnsName, false), position),
                                    RequestedSigner = sd.RequestedSigner,
                                    SignerId = _proxy.GetSignerId(files, sd),
                                    CanUserSign = false,
                                    IsSigned = true,
                                    IsChecking = true,
                                    IsValid = certificateChecker.CheckSignature(signature),
                                    SignId = sd.SpotId,
                                    Role = sd.Intent,
                                    SignDate = signature.SigningTime.ToLocalTime().ToString("g"),
                                    IsCertificateValid = certificateChecker.CheckCertificate(certificate2)
                                });
                            }
                            else
                            {
                                signatureInfoList.Add(new SignatureDisplayParams
                                {
                                    Certificate = null,
                                    SignerName = _proxy.GetPersonTitleForDefinition(files, sd),
                                    RequestedSigner = sd.RequestedSigner,
                                    SignerId = _proxy.GetSignerId(files, sd),
                                    CanUserSign = _proxy.CanSign(files, sd),
                                    IsSigned = false,
                                    IsValid = false,
                                    IsChecking = false,
                                    SignId = sd.SpotId,
                                    Role = sd.Intent
                                });
                            }
                        }

                        // Список подписей, которые содержатся в документе БЕЗ запросов:
                        signatures.ForEach(s =>
                        {
                            var certificate2 = new X509Certificate2(s.SignerCertificate);
                            var position = XpsSignatureUtils.GetPositionFromCertificate(certificate2.Subject);
                            signatureInfoList.Add(new SignatureDisplayParams
                            {
                                Certificate = certificate2,
                                SignerName = _proxy.GetPersonTitleForSignature(files, null, certificate2.GetNameInfo(X509NameType.DnsName, false), position),
                                CanUserSign = false,
                                IsSigned = true,
                                IsChecking = true,
                                SignDate = s.SigningTime.ToLocalTime().ToString("g"),
                                IsValid = certificateChecker.CheckSignature(s),
                                IsCertificateValid = certificateChecker.CheckCertificate(certificate2),
                                SignId = s.Id
                            });
                        });
                    }
                }
                return signatureInfoList;
            }
            catch (Exception ex)
            {
                throw new SignatureException("Error reading signatures", ex);
            }
        }


        public Collection<SignatureDisplayParams> SignAs(INObject document,  IList<SignatureDisplayParams> signatureDisplayParams, DocumentContext context, X509Certificate2 cert, bool actualizeSignatures)
        {
            //if (actualizeSignatures)
            //     _proxy.ActualizeSignaturesInObject();

            var stream = context.ModifyStream(DocumentUsageArea.Signatures);
            var res = XpsSignatureHelper.SignRequests(document, signatureDisplayParams, stream, cert, _proxy);
            stream.Register();
            return res;
        }
    }

    [Serializable]
    public class CertificateNotFoundException : ContextException
    {
        public CertificateNotFoundException(string message, string processName)
            : base(message, processName)
        {
        }
    }
}
