/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.IO.Packaging;
using System.Linq;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.Text.RegularExpressions;
using System.Windows.Xps;
using System.Windows.Xps.Packaging;
using System.Xml;
using System.Xml.Linq;
using Pilot.Xps.Entities;

namespace Pilot.Xps.Domain.Signatures
{
    static class XpsSignatureUtils
    {
        private enum XpsVersion
        {
            Xps,
            Oxps
        }

        /// <summary>
        /// Разбор XPS документа
        /// </summary>
        /// <returns>Полученные из XPS ЭЦП</returns>
        public static List<DigitalSignature> GetDigitalSignatures(string xpsFile)
        {
            using (var p = Package.Open(xpsFile))
                return GetDigitalSignatures(p);
        }

        public static bool HasSignature(Stream stream)
        {
            using (var package = Package.Open(stream))
                return GetDigitalSignatures(package).Any();
        }

        /// <summary>
        /// Разбор XPS документа
        /// </summary>
        /// <returns>Полученные из XPS ЭЦП</returns>
        public static List<DigitalSignature> GetDigitalSignatures(Package package)
        {
            var signatures = new List<DigitalSignature>();
            var signOriginRel = package.GetRelationshipsByType(XpsConstants.TYPE_DIGITAL_SIGNATURE_ORIGIN).FirstOrDefault();
            if (signOriginRel == null)
                return signatures;

            var signOriginPart = package.GetPart(signOriginRel.TargetUri.Normalize());
            var signRels = signOriginPart.GetRelationshipsByType(XpsConstants.TYPE_DIGITAL_SIGNATURE);
            foreach (var signRel in signRels)
            {
                var signPart = package.GetPart(signRel.TargetUri.Normalize());

                using (var destinationStream = new MemoryStream())
                {
                    using (Package destinationPackage = Package.Open(destinationStream, FileMode.Create))
                    {
                        CopyPartToPackage(signPart, destinationPackage);

                        var certRels = signPart.GetRelationshipsByType(XpsConstants.TYPE_CERTIFICATE);
                        foreach (var certRel in certRels)
                        {
                            var certPart = package.GetPart(certRel.TargetUri.Normalize());
                            CopyPartToPackage(certPart, destinationPackage);

                            var relationshipSource = destinationPackage.GetPart(certRel.SourceUri);
                            relationshipSource.CreateRelationship(certPart.Uri, TargetMode.Internal, XpsConstants.TYPE_CERTIFICATE);
                        }
                    }
                    signatures.Add(new DigitalSignature(destinationStream.ToArray()));
                }
            }

            return signatures;
        }

        /// <summary>
        /// Добавление в XPS списка ЭЦП
        /// </summary>
        public static SignResult SignPackage(Package package, IList<DigitalSignature> digitalSignatures)
        {
            SignResult res = 0;
            foreach (var digitalSignature in digitalSignatures)
            {
                res |= Sign(package, digitalSignature);
            }
            return res;
        }

        public static bool IsResultOk(this SignResult result)
        {
            return (~(SignResult.SignedSuccessfully | SignResult.SignatureExists) & result) == 0;
        }

        /// <summary>
        /// Извлечение сертификата из ЭЦП
        /// </summary>
        /// <param name="digitalSignature"></param>
        /// <returns></returns>
        public static X509Certificate GetCertificateFromPackage(DigitalSignature digitalSignature)
        {
            using (var signatureStream = new MemoryStream(digitalSignature.Data))
            {
                using (var signPackage = Package.Open(signatureStream, FileMode.Open, FileAccess.Read))
                {
                    var certPart = signPackage.GetParts().First(p => Path.GetExtension(p.Uri.OriginalString) == XpsConstants.CERTIFICATE_PART_EXTENSION);
                    using (var stream = certPart.GetStream())
                    {
                        using (var ms = new MemoryStream())
                        {
                            stream.CopyTo(ms);
                            return new X509Certificate(ms.ToArray());
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Добавление в XPS документ ЭЦП
        /// </summary>
        /// <param name="package">Пакет XPS документа</param>
        /// <param name="signature">Подпись</param>
        /// <returns>Результат добавления подписи</returns>
        private static SignResult Sign(Package package, DigitalSignature signature)
        {
            using (var signatureStream = new MemoryStream(signature.Data))
            using (var signaturePackage = Package.Open(signatureStream, FileMode.Open, FileAccess.Read))
            {
                if (SignatureExists(package, signaturePackage))
                    return SignResult.SignatureExists;
                //TODO: проверка  на соответствие запросов и подписей ненужна, потому что теперь существуют подписи в nFile'ах без запросов

                if (!CheckForPackageConflicts(package, signaturePackage))
                    return SignResult.PackageConflict;

                var signOriginRel = package.GetRelationshipsByType(XpsConstants.TYPE_DIGITAL_SIGNATURE_ORIGIN).FirstOrDefault();
                if (signOriginRel == null)
                {
                    Uri signOriginUri = new Uri(XpsConstants.SIGNATURE_ORIGIN_URI_STRING, UriKind.Relative).Normalize();
                    package.CreatePart(signOriginUri, XpsConstants.SIGNATURE_ORIGIN_CONTENT_TYPE);
                    signOriginRel = package.CreateRelationship(signOriginUri, TargetMode.Internal, XpsConstants.TYPE_DIGITAL_SIGNATURE_ORIGIN);
                }
                var signOriginPart = package.GetPart(signOriginRel.TargetUri.Normalize());
                foreach (var signaturePackagePart in signaturePackage.GetParts())
                {
                    if (signaturePackagePart.IsRelationship())
                        continue;

                    CopyPartToPackage(signaturePackagePart, package);
                    foreach (var relationship in signaturePackagePart.GetRelationships())
                    {
                        var part = package.GetPart(signaturePackagePart.Uri);
                        part.CreateRelationship(relationship.TargetUri, relationship.TargetMode, relationship.RelationshipType);
                    }
                }
                var digitalSignaturePart = signaturePackage.GetParts().First(IsDigitalSignature);
                signOriginPart.CreateRelationship(digitalSignaturePart.Uri, TargetMode.Internal, XpsConstants.TYPE_DIGITAL_SIGNATURE);
            }
            return SignResult.SignedSuccessfully;
        }

        private static XpsVersion DetermineXpsVersion(Package package)
        {
            var documentSequenceRels = package.GetRelationshipsByType("http://schemas.microsoft.com/xps/2005/06/fixedrepresentation");
            if (documentSequenceRels.Any())
                return XpsVersion.Xps;
            documentSequenceRels = package.GetRelationshipsByType("http://schemas.openxps.org/oxps/v1.0/fixedrepresentation");
            if (documentSequenceRels.Any())
                return XpsVersion.Oxps;

            throw new NotSupportedException();
        }

        private static bool CheckForPackageConflicts(Package package, Package signaturePackage)
        {
            foreach (var signaturePackagePart in signaturePackage.GetParts())
            {
                if (signaturePackagePart.IsRelationship())
                    continue;

                if (package.PartExists(signaturePackagePart.Uri))
                {
                    using (var stream1 = package.GetPart(signaturePackagePart.Uri).GetStream())
                    using (var stream2 = signaturePackagePart.GetStream())
                    {
                        if (ComputeMd5(stream1) != ComputeMd5(stream2))
                        {
                            return false;
                        }
                    }
                }
            }
            return true;
        }

        private static XpsConstants CreateConstants(XpsVersion version)
        {
            switch (version)
            {
                case XpsVersion.Xps:
                    return new XpsConstants();
                case XpsVersion.Oxps:
                    return new OxpsConstants();
                default: throw new NotSupportedException();
            }
        }

        public static bool HasSignatureRequests(Package package)
        {
            var constants = new XpsConstants();
            var documentSequenceRels = package.GetRelationshipsByType(constants.TypeFixedPresentation);
            foreach (var documentSequenceRel in documentSequenceRels)
            {
                var documentSequencePart = package.GetPart(documentSequenceRel.TargetUri.Normalize());
                using (var documentSequenceStream = documentSequencePart.GetStream())
                {
                    var documentSequenceXmlRoot = XElement.Load(documentSequenceStream);
                    var documentReferences = documentSequenceXmlRoot.Descendants(XName.Get(XpsConstants.TAG_DOCUMENT_REFERENCE, constants.XpsXmlns));
                    foreach (var documentReference in documentReferences)
                    {
                        var sourceAttribute = documentReference.Attribute(XpsConstants.SOURCE_ATTRIBUTE_NAME);
                        if (sourceAttribute == null)
                            continue;

                        var documentReferenceSourceUri = new Uri(sourceAttribute.Value, UriKind.Relative).Normalize();
                        var documentPart = package.GetPart(documentReferenceSourceUri);
                        var signatureDefinitionRel = documentPart.GetRelationshipsByType(constants.TypeSignatureDefinitions).FirstOrDefault();
                        if (signatureDefinitionRel != null)
                        {
                            return true;
                        }
                    }
                }
            }
            return false;
        }

        /// <summary>
        /// Получение ID запросов на подпись в XPS документе
        /// </summary>
        /// <returns>Список ID запросов на подпись</returns>
        private static IEnumerable<Guid> GetSignatureRequests(Package package, XpsConstants constants)
        {
            var documentSequenceRels = package.GetRelationshipsByType(constants.TypeFixedPresentation);
            foreach (var documentSequenceRel in documentSequenceRels)
            {
                var documentSequencePart = package.GetPart(documentSequenceRel.TargetUri.Normalize());
                using (var documentSequenceStream = documentSequencePart.GetStream())
                {
                    var documentSequenceXmlRoot = XElement.Load(documentSequenceStream);
                    var documentReferences = documentSequenceXmlRoot.Descendants(XName.Get(XpsConstants.TAG_DOCUMENT_REFERENCE, constants.XpsXmlns));
                    foreach (var documentReference in documentReferences)
                    {
                        var sourceAttribute = documentReference.Attribute(XpsConstants.SOURCE_ATTRIBUTE_NAME);
                        if (sourceAttribute == null)
                            continue;

                        Uri documentReferenceSourceUri = new Uri(sourceAttribute.Value, UriKind.Relative).Normalize();
                        var documentPart = package.GetPart(documentReferenceSourceUri);
                        var signatureDefinitionRel = documentPart.GetRelationshipsByType(constants.TypeSignatureDefinitions).FirstOrDefault();
                        if (signatureDefinitionRel != null)
                        {
                            var signatureDefinitionPart = package.GetPart(signatureDefinitionRel.TargetUri.Normalize());
                            using (var signatureDefinitionStream = signatureDefinitionPart.GetStream())
                            {
                                var signatureDefinitionXmlRoot = XElement.Load(signatureDefinitionStream);
                                var signatureDefinitions = signatureDefinitionXmlRoot.Descendants(XName.Get(XpsConstants.TAG_SIGNATURE_DEFINITION, constants.TypeSignatureDefinitions));
                                foreach (var signatureDefinition in signatureDefinitions)
                                {
                                    var spotIdAttribute = signatureDefinition.Attribute(XpsConstants.SPOT_ID_ATTRIBUTE_NAME);
                                    if (spotIdAttribute != null)
                                    {
                                        Guid value;
                                        try
                                        {
                                            value = new Guid(spotIdAttribute.Value);
                                        }
                                        catch (FormatException)
                                        {
                                            value = Guid.Empty;
                                        }
                                        yield return value;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Получение ID подписи (должно совпадать с соответствующим запросом на подпись)
        /// </summary>
        public static Guid GetSignatureId(Package signaturePackage)
        {
            var signaturePart = signaturePackage.GetParts().First(IsDigitalSignature);
            using (var stream = signaturePart.GetStream())
            {
                var signature = XElement.Load(stream);
                var idAttribute = signature.Attribute(XName.Get("Id"));
                var signatureGuid = Guid.Empty;
                if (idAttribute != null)
                {
                    Guid.TryParse(XmlConvert.DecodeName(idAttribute.Value), out signatureGuid);
                }
                return signatureGuid;
            }
        }

        private static bool SignatureExists(Package xpsPackage, Package signaturePackage)
        {
            var signaturePart = signaturePackage.GetParts().First(IsDigitalSignature);
            return xpsPackage.PartExists(signaturePart.Uri);
        }

        private static bool IsDigitalSignature(PackagePart part)
        {
            return Path.GetExtension(part.Uri.OriginalString) == XpsConstants.SIGNATURE_PART_EXTENSION;
        }

        private static void CopyPartToPackage(PackagePart part, Package package)
        {
            if (package.PartExists(part.Uri))
                return;

            var destinationPart = package.CreatePart(part.Uri, part.ContentType);
            Debug.Assert(destinationPart != null, "destinationPart != null");
            using (var destinationStream = destinationPart.GetStream())
            using (var sourceStream = part.GetStream())
            {
                sourceStream.CopyTo(destinationStream);
            }
        }

        private static Uri Normalize(this Uri uri)
        {
            if (uri.OriginalString.StartsWith("/"))
                return uri;
            return new Uri("/" + uri.OriginalString, UriKind.Relative);
        }

        static string ComputeMd5(Stream stream)
        {
            stream.Position = 0;
            using (var md5Hasher = new MD5CryptoServiceProvider())
            {
                var data = md5Hasher.ComputeHash(stream);
                var result = BitConverter.ToString(data).Replace("-", String.Empty).ToLower();
                return result;
            }
        }

        private static bool IsRelationship(this PackagePart part)
        {
            return part.ContentType == XpsConstants.RELATIONSHIP_CONTENT_TYPE;
        }

        //public static Guid NewSpotId()
        //{
        //    return Common.Utils.SignatureUtils.NewSpotId();
        //}

        public static Guid Id2Guid(Guid databaseId, int id)
        {
            var intBytes = BitConverter.GetBytes(id);
            var sGuid = "ffffffff" + databaseId.ToString().Substring(8);
            var array = new Guid(sGuid).ToByteArray();
            array[4] = intBytes[3];
            array[5] = intBytes[2];
            array[6] = intBytes[1];
            array[7] = intBytes[0];
            return new Guid(array);
        }

        public static int Guid2Id(Guid? guid)
        {
            if (guid != null)
            {
                var array = ((Guid)guid).ToByteArray();
                return (int)array[7] + ((int)array[6] << 8) + ((int)array[5] << 16) + ((int)array[4] << 24);
            }
            return 0;
        }

        public static bool IsGuidFromCurrentBase(Guid databaseId, Guid? guid)
        {
            if (guid != null)
            {
                var guidArray = ((Guid)guid).ToByteArray();
                var databaseIdArray = databaseId.ToByteArray();
                for (int i = 8; i < 16; i++)
                    if (guidArray[i] != databaseIdArray[i])
                        return false;
                return true;
            }
            return false;
        }

        public static void RemoveSignatureDefinitions(this XpsDocument xpsDoc)
        {
            var fixedDoc = xpsDoc.FixedDocumentSequenceReader.FixedDocuments[0];
            var storedDefinitions = fixedDoc.GetSignatureDefinitions();
            if (storedDefinitions.Count == 0)
                return;

            var definitionsToRemove = new XpsSignatureDefinition[storedDefinitions.Count];
            storedDefinitions.CopyTo(definitionsToRemove, 0);
            foreach (var d in definitionsToRemove)
            {
                fixedDoc.RemoveSignatureDefinition(d);
            }
            fixedDoc.CommitSignatureDefinition();
        }

        public static ICollection<XpsSignatureDefinition> GetSignatureDefinitions(this IXpsFixedDocumentReader fixedDoc)
        {
            try
            {
                return fixedDoc.SignatureDefinitions;
            }
            catch (XpsPackagingException)
            {
                // не понятно, почему при первом обращении кидается Exception
                try
                {
                    return fixedDoc.SignatureDefinitions;
                }
                catch (XpsPackagingException)
                {
                    return new Collection<XpsSignatureDefinition>();
                }
            }
        }

        public static ICollection<XpsSignatureDefinition> GetSignatureDefinitions(this XpsDocument xpsDoc)
        {
            return xpsDoc.FixedDocumentSequenceReader.FixedDocuments[0].GetSignatureDefinitions();
        }

        public static void AddDefinitions(this XpsDocument xpsDoc, XpsSignatureDefinition[] signatureDefinitions)
        {
            if (signatureDefinitions.Length == 0)
                return;
            var fixedDoc = xpsDoc.FixedDocumentSequenceReader.FixedDocuments[0];
            fixedDoc.GetSignatureDefinitions();
            foreach (var sd in signatureDefinitions)
            {
                if (sd.Intent == null)
                    sd.Intent = "";
                if (sd.SpotId == null)
                    sd.SpotId = Guid.Empty;

                fixedDoc.AddSignatureDefinition(sd);
            }
            fixedDoc.CommitSignatureDefinition();
        }

        public static string GetPositionFromCertificate(string certificateSubject)
        {
            var position = "";
            var match = Regex.Match(certificateSubject, @"(?<=\b, T=).+?(?=,)");
            if (match.Success)
                position = match.Value;
            return position;
        }
    }
}