/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Globalization;
using System.IO;
using System.IO.Packaging;
using System.Linq;
using System.Windows;
using System.Windows.Markup;
using System.Xml;
using System.Xml.Linq;
using Pilot.Xps.Domain.Labels;
using Path = System.Windows.Shapes.Path;

namespace Pilot.Xps.Domain
{
    class XpsTools
    {
        public static Stream GetAnnotationMessagesStreamInPackage(Package package, bool isCreateMode)
        {
            return XpsTools.GetStreamInPackage(
                package, 
                XpsConstants.ANNOTATION_MESSAGES_RELS_TYPE,
                XpsConstants.ANNOTATION_MESSAGES_DATA_FILE, 
                XpsConstants.ANNOTATION_MESSAGES_CONTENT_TYPE,
                isCreateMode
                );
        }

        public static Stream GetStreamInPackage(Package package, string relsType, string dataFile, string contentType, bool isCreateMode)
        {
            var packagePart = GetDocumentPart(package);
            PackageRelationship packageRelationship = null;
            var fileMode = isCreateMode ? FileMode.Create : FileMode.Open;
            foreach (var rel in packagePart.GetRelationshipsByType(relsType))
            {
                packageRelationship = rel;
            }

            PackagePart resultPackagePart;
            if (packageRelationship == null)
            {
                if (isCreateMode)
                {
                    resultPackagePart = package.CreatePart(PackUriHelper.CreatePartUri(new Uri(dataFile, UriKind.Relative)), contentType);
                    if (resultPackagePart != null)
                        packagePart.CreateRelationship(resultPackagePart.Uri, TargetMode.Internal, relsType);
                }
                else
                {
                    return null;
                }
            }
            else
            {
                resultPackagePart = package.GetPart(packageRelationship.TargetUri);
            }

            var textBlockStream = resultPackagePart?.GetStream(fileMode);
            return textBlockStream;
        }

        public static BarcodeDescription GetBarcode(string barcodeConfiguration)
        {
            var barcode = TryParseXml(barcodeConfiguration);
            if (barcode == null)
                return null;

            var result = new BarcodeDescription();

            var angleAttr = barcode.Attribute(BarcodeLabelsConstants.ANGLE_ATTRIBUTE);
            var angle = angleAttr?.Value ?? "0";
            result.Angle = Convert.ToDouble(angle, CultureInfo.InvariantCulture);

            var pageRangeAttr = barcode.Attribute(BarcodeLabelsConstants.TEXT_LABEL_PAGE_RANGE);
            var pageRange = pageRangeAttr?.Value ?? "1";
            result.PageRange = pageRange;

            var scaleAttr = barcode.Attribute(BarcodeLabelsConstants.SCALE_ATTRIBUTE);
            var scale = scaleAttr?.Value ?? "0";
            result.Scale = Convert.ToDouble(scale, CultureInfo.InvariantCulture);

            var vertAligmentAttr = barcode.Attribute(BarcodeLabelsConstants.VERTICAL_ALIGNMENT);
            var vertAligment = vertAligmentAttr?.Value ?? "";
            VerticalAlignment verticalAlignment;
            Enum.TryParse(vertAligment, out verticalAlignment);
            result.VerticalAlignment = verticalAlignment;

            var horAligmentAttr = barcode.Attribute(BarcodeLabelsConstants.HORIZONTAL_ALIGNMENT);
            var horAligment = horAligmentAttr?.Value ?? "";
            HorizontalAlignment horizontalAlignment;
            Enum.TryParse(horAligment, out horizontalAlignment);
            result.HorizontalAlignment = horizontalAlignment;

            var textElement = barcode.Elements(BarcodeLabelsConstants.TEXT_BLOCK).FirstOrDefault();
            if (textElement != null)
            {
                if (string.IsNullOrEmpty(textElement.Value))
                    textElement.Value = string.Empty;
                var textLabel = new TextLabel(textElement.ToString());
                var textBlock = BarcodeLabelsDrawer.DrawTextBlock(textLabel);
                result.TextBlock = textBlock;
            }

            var pathElement = barcode.Elements(XpsConstants.PATH).FirstOrDefault();
            if (pathElement != null)
            {
                if (string.IsNullOrEmpty(pathElement.Value))
                    pathElement.Value = string.Empty;
                result.Barcode = ParsePathToObject(pathElement.ToString());
                var barcodeOffsetX = pathElement.Attribute(BarcodeLabelsConstants.CANVAS_LEFT);
                result.BarcodeOffsetX = barcodeOffsetX != null ? barcodeOffsetX.Value : "0";

                var barcodeOffsetY = pathElement.Attribute(BarcodeLabelsConstants.CANVAS_TOP);
                result.BarcodeOffsetY = barcodeOffsetY != null ? barcodeOffsetY.Value : "0";
            }

            result.IsFloating = ((bool?)(barcode.Attribute(BarcodeLabelsConstants.FLOATING_ATTRIBUTE))).GetValueOrDefault();

            return result;
        }

        public static Stream GetStreamInPackageAllowingMultipleFiles(Package package, string relsType, string dataFile, string contentType)
        {
            var packagePart = GetPagePart(package);
            PackagePart resultPackagePart = null;

            resultPackagePart =
                package.CreatePart(PackUriHelper.CreatePartUri(new Uri(dataFile, UriKind.Relative)), contentType);
            if (resultPackagePart != null)
                packagePart.CreateRelationship(resultPackagePart.Uri, TargetMode.Internal, relsType);

            var textBlockStream = resultPackagePart.GetStream(FileMode.Create);
            return textBlockStream;
        }

        public static void DeletePartAndRelationship(Package package, string dataFile, string relsType)
        {
            package.DeletePart(new Uri(dataFile, UriKind.Relative));
            var documentPart = GetDocumentPart(package);
            var relation = documentPart.GetRelationshipsByType(relsType).FirstOrDefault();
            if (relation != null)
                documentPart.DeleteRelationship(relation.Id);
        }

        internal static PackagePart GetDocumentPart(Package package)
        {
            var rootUri = (from part in package.GetParts() where part.ContentType == XpsConstants.FIXED_DOCUMENT_SEQUENCE_CONTENT_TYPE select part.Uri).FirstOrDefault();
            if (rootUri == null)
            {
                throw new InvalidOperationException("Invalid document file");
            }

            var docPart = package.GetPart(rootUri);
            return docPart;
        }
        internal static PackagePart GetPagePart(Package package)
        {
            var rootUri =
                (from part in package.GetParts()
                    where part.ContentType == XpsConstants.FIXED_PAGE_CONTENT_TYPE
                    select part.Uri).
                FirstOrDefault();

            if (rootUri == null)
            {
                throw new InvalidOperationException("Invalid document file");
            }

            var docPart = package.GetPart(rootUri);
            return docPart;
        }

        public static XElement TryParseXml(string xml)
        {
            try
            {
                return string.IsNullOrEmpty(xml) ? null : XElement.Parse(xml);
            }
            catch (Exception)
            {
                return null;
            }
        }

        public static bool IsDataFileExist(Package package, string relsType, string dataFile, string contentType)
        {
            var packagePart = GetDocumentPart(package);
            PackageRelationship packageRelationship = null;
            foreach (var rel in packagePart.GetRelationshipsByType(relsType))
            {
                packageRelationship = rel;
            }
            return packageRelationship != null;
        }

        private static Path ParsePathToObject(string textBlock)
        {
            if (string.IsNullOrEmpty(textBlock))
                return null;

            try
            {
                var nsmgr = new XmlNamespaceManager(new NameTable());
                nsmgr.AddNamespace(String.Empty, @"http://schemas.microsoft.com/winfx/2006/xaml/presentation");
                var xmlParserContext = new XmlParserContext(null, nsmgr, null, XmlSpace.None);
                var reader = XmlReader.Create(new StringReader(textBlock), null, xmlParserContext);
                return XamlReader.Load(reader) as Path;
            }
            catch
            {
                return null;
            }
        }
    }
}
