/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Concurrent;
using Ascon.Pilot.Server.Api.Contracts;
using Ascon.Pilot.Transport;

namespace Pilot.Xps.Connection.Server
{
    interface IRemoteCommandsApi
    {
        void RegisterCommand(string commandName, CommandRequestedDelegate callback);
        void RegisterCommandHandler<T>(T handler);
    }

    delegate void CommandRequestedDelegate(string commandName, Guid requestId, byte[] data);

    class RemoteCommandsApi : IRemoteCommandsApi, IServerCommandCallback
    {
        private readonly string _commandProcessorName;
        private readonly IServerCommandApi _serverCommandApi;
        private readonly ConcurrentDictionary<Type, CommandRequestedDelegate> _commandListeners = new ConcurrentDictionary<Type, CommandRequestedDelegate>();
        private readonly ConcurrentDictionary<string, CommandRequestedDelegate> _commandByNameListeners = new ConcurrentDictionary<string, CommandRequestedDelegate>();

        public RemoteCommandsApi(Func<IServerCommandCallback, IServerCommandApi> serverCommandApiFunc, string commandProcessorName)
        {
            _commandProcessorName = commandProcessorName;
            _serverCommandApi = serverCommandApiFunc(this);
        }

        public void RegisterCommandHandler<T>(T handler)
        {
            var marshaller = new ServerCommandMarshaller<T>(handler,
                _serverCommandApi.RegisterCommandHandler,
                (id, data) => _serverCommandApi.PublishCommandResult(id, data, ServerCommandResult.Success),
                (id, data) => _serverCommandApi.PublishCommandResult(id, data, ServerCommandResult.Error),
                _commandProcessorName);
            _commandListeners[typeof(T)] = marshaller.CommandRequested;
        }

        public void RegisterCommand(string commandName, CommandRequestedDelegate callback)
        {
            _serverCommandApi.RegisterCommandHandler(commandName);
            _commandByNameListeners[commandName] = callback;
        }

        public void NotifyCommandRequested(string commandName, Guid requestId, byte[] data)
        {
            foreach (var listener in _commandListeners.Values)
            {
                listener(commandName, requestId, data);
            }

            if (_commandByNameListeners.TryGetValue(commandName, out var callback))
            {
                callback(commandName, requestId, data);
            }
        }
    }
}
