/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.Collections.Generic;
using System.Security;
using System.Security.Cryptography.X509Certificates;
using Ascon.Pilot.Common;
using Ascon.Pilot.Common.Connection;
using Ascon.Pilot.Common.DataProtection;
using Pilot.Xps.Connection;

namespace Pilot.Xps.Service
{
    class AppSettings : IAppSettings
    {
        public AppSettings(AppSettingsJson json)
        {
            ConnectionSettings = new ConnectionSettings
            {
                LicenseCode = json.Connection.LicenseCode,
                ConnectionParams = new ConnectionParams
                {
                    Server = json.Connection.Server,
                    UserName = json.Connection.UserName,
                    Password = GetPassword(json.Connection.Password)
                }
            };
            SigningSettings = new SigningSettings
            {
                StoreLocation = json.Signing.StoreLocation,
                CertificatesMap = new Dictionary<string, string>(json.Signing.CertificatesMap)
            };
        }

        public IConnectionSettings ConnectionSettings { get; set; }
        public ISigningSettings SigningSettings { get; }

        private SecureString GetPassword(string plainPassword)
        {
            if (string.IsNullOrEmpty(plainPassword))
                return new SecureString();
            try
            {
                return plainPassword.DecryptAes().ConvertToSecureString();
            }
            catch
            {
                return new SecureString();
            }
            
        }
    }

    public class ConnectionSettings : IConnectionSettings
    {
        public int LicenseCode { get; set; }
        public ConnectionParams ConnectionParams { get; set; }
    }

    public class SigningSettings : ISigningSettings
    {
        public Dictionary<string, string> CertificatesMap { get; set; }
        public StoreLocation StoreLocation { get; set; }
    }

    public class AppSettingsJson
    {
        public ConnectionJson Connection { get; set; }
        public SigningSettingsJson Signing { get; set; }
    }

    public class ConnectionJson
    {
        public int LicenseCode { get; set; }
        public string Server { get; set; }
        public string UserName { get; set; }
        public string Password { get; set; }
    }

    public class SigningSettingsJson
    {
        public StoreLocation StoreLocation { get; set; }
        public Dictionary<string, string> CertificatesMap { get; set; }
    }
}
