/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using PilotRocketChatGateway.Authentication;
using PilotRocketChatGateway.PilotServer;
using PilotRocketChatGateway.UserContext;
using PilotRocketChatGateway.Utils;

namespace PilotRocketChatGateway.Controllers
{
    [ApiController]
    public class ChatController : ControllerBase
    {
        private readonly IContextsBank _contextsBank;
        private readonly IAuthHelper _authHelper;

        public ChatController(IContextsBank contextsBank, IAuthHelper authHelper)
        {
            _contextsBank = contextsBank;
            _authHelper = authHelper;
        }


        [Authorize]
        [HttpGet("api/v1/chat.search")]
        public string Search(string roomId, string searchText, int count, int offset)
        {
            var context = _contextsBank.GetContext(HttpContext.GetTokenActor(_authHelper));
            var (msgs, total) = context.ChatService.DataLoader.SearchMessages(roomId, searchText, count, offset);
            var result = new
            {
                messages = msgs,
                success = true
            };
            return JsonConvert.SerializeObject(result);
        }

        [Authorize]
        [HttpGet("api/v1/chat.syncMessages")]
        public string Get() 
        {
            var roomId = HttpContext.Request.Query["roomId"];
            var lastUpdate = HttpContext.Request.Query["lastUpdate"];
            if (!string.IsNullOrEmpty(lastUpdate))
                return OldSyncMessages(roomId, lastUpdate);

            var type = HttpContext.Request.Query["type"];

            var nextString = HttpContext.Request.Query["next"];
            long next = string.IsNullOrEmpty(nextString) ? 0 : long.Parse(nextString);

            var previousString = HttpContext.Request.Query["next"];
            long previous = string.IsNullOrEmpty(previousString) ? 0 : long.Parse(previousString);

            var countString = HttpContext.Request.Query["count"];
            int count = string.IsNullOrEmpty(countString) ? 0 : int.Parse(countString);

            var offsetString = HttpContext.Request.Query["offset"];
            int offset = string.IsNullOrEmpty(offsetString) ? 0 : int.Parse(offsetString);

            IList<Message> msgs = new List<Message>();
            if (type == "UPDATED")
            {
                var context = _contextsBank.GetContext(HttpContext.GetTokenActor(_authHelper));

                if (next != 0)
                    msgs = context.ChatService.DataLoader.LoadPreviousMessages(roomId, count, offset, next);
                else if (previous != 0)
                    msgs = context.ChatService.DataLoader.LoadMessages(roomId, count, offset, previous);
                else
                    throw new ArgumentException("incorrect parametres for chat.syncMessages endpoint");
            }

            var res = new
            {
                result = new
                {
                    updated = msgs,
                    deleted = new List<Message>(),
                },
                success = true
            };
            return JsonConvert.SerializeObject(res);
        }

        [Authorize]
        [HttpPost("api/v1/chat.sendMessage")]
        public string SendMessage(object request)
        {
            var message = JsonConvert.DeserializeObject<MessageRequest>(request.ToString()).message;
            var context = _contextsBank.GetContext(HttpContext.GetTokenActor(_authHelper));
            var (link, text) = MarkdownHelper.CutHyperLink(message.msg);
            context.ChatService.DataSender.SendTextMessageToServer(message.roomId, message.id, text, link);
            var result = new MessageRequest()
            {
                success = true
            };
            return JsonConvert.SerializeObject(result);
        }

        [Authorize]
        [HttpPost("api/v1/chat.update")]
        public string Update(object request)
        {
            var message = JsonConvert.DeserializeObject<MessageEdit>(request.ToString());
            var context = _contextsBank.GetContext(HttpContext.GetTokenActor(_authHelper));
            context.ChatService.DataSender.SendEditMessageToServer(message.roomId, message.msgId, message.text);
            var result = new MessageRequest()
            {
                success = true
            };
            return JsonConvert.SerializeObject(result);
        }

        [Authorize]
        [HttpGet("api/v1/chat.getMessage")]
        public string GetMessage(string msgId)
        {
            var context = _contextsBank.GetContext(HttpContext.GetTokenActor(_authHelper));
            var msg = context.ChatService.DataLoader.LoadMessage(msgId);
            var res = new
            {
                message = msg,
                success = true
            };
            return JsonConvert.SerializeObject(res);

        }

        private string OldSyncMessages(string roomId, string lastUpdate)
        {
            var context = _contextsBank.GetContext(HttpContext.GetTokenActor(_authHelper));
            var msgs = context.ChatService.DataLoader.LoadMessages(roomId, lastUpdate);
            var res = new
            {
                result = new
                {
                    updated = msgs,
                    deleted = new List<Message>(),
                },
                success = true
            };
            return JsonConvert.SerializeObject(res);
        }
    }
}
