/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Primitives;
using Newtonsoft.Json;
using System.Reflection;
using System.Security.Cryptography;

namespace PilotRocketChatGateway.Controllers
{
    [ApiController]
    public class SettingsController : ControllerBase
    {
        private const string ROCKET_CHAT_SETTNGS_FILE = "rocketchatsettings.json";
        private static Dictionary<string, Setting> _rocketChatSettings { get; } = LoadRocketChatSettings();

        [HttpGet("api/v1/settings.oauth")]
        public OauthSettings Outh()
        {
            return new OauthSettings { success = false, services = new string[] { } };
        }

        [HttpGet("api/v1/settings.public")]
        public string Get()
        {
            try
            {
                var queryParameter = HttpContext.Request.Query.First();
                if (queryParameter.Key == "query")
                    return OldApi(queryParameter);

                var settings = queryParameter.Value.ToString().Split(",");
                var serverSettings = GetServerSetting(settings);
                return JsonConvert.SerializeObject(serverSettings);
            }
            catch (Exception)
            {
                var settings = new SettingsListRequest()
                {
                    settings = new List<string>()
                };
                return JsonConvert.SerializeObject(settings);
            }
        }

        private string OldApi(KeyValuePair<string, StringValues> queryParameter)
        {
            var settings = JsonConvert.DeserializeObject<SettingsRequest>(queryParameter.Value.ToString());
            var serverSettings = GetServerSetting(settings.settings.settings.ToArray());
            return JsonConvert.SerializeObject(serverSettings);
        }

        private ServerSettings GetServerSetting(string[] settings)
        {
            var result = new ServerSettings
            {
                success = true,
                settings = new List<Setting>(),
                total = settings.Count()
            };

            foreach (var setting in settings)
            {
                if (_rocketChatSettings.TryGetValue(setting, out var val))
                    result.settings.Add(val);
            }

            return result;
        }

        private static Dictionary<string, Setting> LoadRocketChatSettings()
        {
            string path = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), ROCKET_CHAT_SETTNGS_FILE);
            string json = System.IO.File.ReadAllText(path);

            var settings = JsonConvert.DeserializeObject<ServerSettings>(json);
            var result = new Dictionary<string, Setting>();
            foreach (var setting in settings.settings)
                result[setting.id] = setting;
            return result;
        }
    }
}
