/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Ascon.Pilot.SDK;
using PilotShare.Client.Utils;
using PilotShare.Client.View;
using PilotShare.Core;
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace PilotShare.Client
{
    public interface ILinkObjectModifier
    {
        void ModifyLink(LinkItem link);
        void CreateLink(LinkItem link, IDataObject target, IContextThreadDispatcher dispatcher);
        void RemoveLink(Guid linkId);
    }

    [Export(typeof(ILinkObjectModifier))]
    public class LinkObjectModifier : ILinkObjectModifier
    {
        private readonly IObjectModifier _modifier;
        private readonly IObjectsRepository _repository;

        [ImportingConstructor]
        public LinkObjectModifier(IObjectModifier modifier, IObjectsRepository repository)
        {
            _modifier = modifier;
            _repository = repository;
        }

        public async void CreateLink(LinkItem link, IDataObject target, IContextThreadDispatcher dispatcher)
        {
            var result = new Item();
            var file = target.ActualFileSnapshot.Files.FirstOrDefault(x => !FileExtensionHelper.IsSystemFile(x.Name));
            result.Name = GetFileSystemName(target, file);

            if (file != null)
            {
                result.FileId = file.Id.ToString();
                result.FileSize = file.Size;
                result.Md5 = file.Md5;
            }

            await TryGetChildren(target, result, dispatcher);
            await CreateLink(link, target, result, dispatcher);
        }

        public void ModifyLink(LinkItem link)
        {
            var editedObject = _modifier.EditById(link.Id);
            ApplyCommonAttributes(link, editedObject);
            _modifier.Apply();
        }

        public void RemoveLink(Guid linkId)
        {
            _modifier.DeleteById(linkId);
            _modifier.Apply();
        }

        private string GetFileSystemName(IDataObject obj, IFile file)
        {
            if (obj.Type.Name == SystemTypeNames.PROJECT_FILE || obj.Type.Name == SystemTypeNames.PROJECT_FOLDER)
                return obj.DisplayName;
            return PathExtensions.RemoveInvalidPathCharacters(obj.DisplayName) + Path.GetExtension(file?.Name);
        }

        private async Task TryGetChildren(IDataObject root, Item rootItem, IContextThreadDispatcher dispatcher)
        {
            if (root.Children.Any())
            {
                var childObjects = await _repository.GetObjectsAsync(root.Children.ToArray(), dispatcher);
                rootItem.Children = new List<Item>();

                foreach (var child in childObjects)
                {
                    var result = new Item();

                    var file = child.ActualFileSnapshot.Files.FirstOrDefault(x =>
                        !FileExtensionHelper.IsSystemFile(x.Name));

                    result.Name = GetFileSystemName(child, file);
                    if (file != null)
                    {
                        result.FileId = file.Id.ToString();
                        result.FileSize = file.Size;
                        result.Md5 = file.Md5;
                    }

                    await TryGetChildren(child, result, dispatcher);
                    rootItem.Children.Add(result);
                }
            }
        }

        private async Task CreateLink(LinkItem link, IDataObject target, Item item, IContextThreadDispatcher dispatcher)
        {
            var externalLinkType = _repository.GetType(Const.EXTERNAL_LINK_TYPE_NAME);
            if (externalLinkType == null)
                return;

            var parent = await _repository.GetObjectAsync(Guid.Empty, dispatcher);
            var newObject = _modifier.Create(link.Id, parent, externalLinkType);

            var currentUser = _repository.GetCurrentPerson();
            newObject.SetAttribute(LinkAttributeNames.AUTHOR, currentUser.Id);
            newObject.SetAttribute(LinkAttributeNames.TARGET, target.Id.ToString());
            
            var data = StructureSerializer.Serialize(item);
            newObject.SetAttribute(LinkAttributeNames.STRUCTURE, data);
            
            ApplyCommonAttributes(link, newObject);
            
            _modifier.Apply();
        }

        private static void ApplyCommonAttributes(LinkItem link, IObjectBuilder builder)
        {
            builder.SetAttribute(LinkAttributeNames.DATE, link.Date.ToUniversalTime());
            builder.RemoveAttribute(LinkAttributeNames.EXPIRATION_DATE);
            if (link.ValidThrough != null)
                builder.SetAttribute(LinkAttributeNames.EXPIRATION_DATE, link.ValidThrough.Value.ToUniversalTime());

            builder.RemoveAttribute(LinkAttributeNames.PASSWORD);
            if (link.Password != null && !string.IsNullOrEmpty(link.Password))
                builder.SetAttribute(LinkAttributeNames.PASSWORD, link.Password);
        }
    }
}
