/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reactive.Concurrency;
using System.Reactive.Disposables;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Threading;

namespace PilotShare.Client.Utils
{
    public interface IContextThreadDispatcher
    {
        void BeginInvoke(Action action);
        void BeginInvoke(Action action, DispatcherPriority priority);
        void Invoke(Action action);
        void Invoke(Action action, DispatcherPriority priority);

        int ThreadId { get; }
    }

    public class ContextDispatcher : IContextThreadDispatcher
    {
        private readonly Dispatcher _dispatcher;

        public ContextDispatcher(Dispatcher dispatcher)
        {
            _dispatcher = dispatcher;
        }

        public void BeginInvoke(Action action)
        {
            _dispatcher.BeginInvoke(action);
        }

        public void BeginInvoke(Action action, DispatcherPriority priority)
        {
            _dispatcher.BeginInvoke(action, priority);
        }

        public void Invoke(Action action)
        {
            _dispatcher.Invoke(action);
        }

        public void Invoke(Action action, DispatcherPriority priority)
        {
            _dispatcher.Invoke(action, priority);
        }

        public int ThreadId => _dispatcher.Thread.ManagedThreadId;
    }

    class DispatcherScheduler : IScheduler
    {
        private readonly IContextThreadDispatcher _dispatcher;
        private readonly DispatcherPriority _priority;

        public DispatcherScheduler(IContextThreadDispatcher dispatcher, DispatcherPriority priority)
        {
            _dispatcher = dispatcher;
            _priority = priority;
        }

        public IDisposable Schedule<TState>(TState state, Func<IScheduler, TState, IDisposable> action)
        {
            if (action == null)
                throw new ArgumentNullException(nameof(action));

            var d = new SingleAssignmentDisposable();

            _dispatcher.BeginInvoke(() =>
            {
                if (d.IsDisposed)
                    return;
                d.Disposable = action((IScheduler)this, state);
            }, _priority);
            return (IDisposable)d;
        }

        public IDisposable Schedule<TState>(TState state, TimeSpan dueTime, Func<IScheduler, TState, IDisposable> action)
        {
            throw new NotImplementedException();
        }

        public IDisposable Schedule<TState>(TState state, DateTimeOffset dueTime, Func<IScheduler, TState, IDisposable> action)
        {
            throw new NotImplementedException();
        }

        public DateTimeOffset Now { get; }
    }
}
