/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reactive.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Threading;
using Ascon.Pilot.SDK;

namespace PilotShare.Client.Utils
{
    static class ObjectsExtensions
    {
        public static Task<IEnumerable<IDataObject>> GetObjectsAsync(this IObjectsRepository repository, Guid[] ids, IContextThreadDispatcher dispatcher, CancellationToken ct = default(CancellationToken))
        {
            // Creates an observable that fires notification when cancelling CancellationToken
            var cancel = Observable.Create<IDataObject>(o => ct.Register(o.OnCompleted));
            var dispatcherScheduler = new DispatcherScheduler(dispatcher, DispatcherPriority.Background);

            var loading = ids.ToList();
            var observableList = repository
                .SubscribeObjects(loading)                          // Subscribing on interested objects
                .TakeUntil(cancel)                                  // Stopping subscription on cancel
                .ObserveOn(dispatcherScheduler)                     // Forcing notifications to be raised on UI thread with Background priority
                .Where(o => o.State == DataState.Loaded)            // Filtering "NoData" notifications
                .Distinct(o => o.Id)                                // Filtering already emitted notifications
                .Take(loading.Count)                                // Wait for all objects to be loaded
                .ToList();

            return Task<IEnumerable<IDataObject>>.Factory.StartNew(() =>
            {
                var lazy = observableList.Wait();
                // Forces the "lazy" enumerable to be immediately iterated in background thread
                return lazy.ToList();
            }, ct);
        }


        public static Task<IDataObject> GetObjectAsync(this IObjectsRepository repository, Guid id, IContextThreadDispatcher dispatcher, CancellationToken ct = default(CancellationToken))
        {
            var cancel = Observable.Create<IDataObject>(o => ct.Register(o.OnCompleted));
            var dispatcherScheduler = new DispatcherScheduler(dispatcher, DispatcherPriority.Background);

            var loading = new[] { id };
            var observableList = repository
                .SubscribeObjects(loading)
                .TakeUntil(cancel)
                .ObserveOn(dispatcherScheduler)
                .Where(o => o.State == DataState.Loaded || o.State == DataState.NonExistent)
                .Distinct(o => o.Id)
                .Take(loading.Length);

            return Task<IDataObject>.Factory.StartNew(() =>
            {
                var dataObject = observableList.Wait();
                return dataObject;
            }, ct);
        }

    }
}
