using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.RazorPages;
using Newtonsoft.Json;
using System.ComponentModel.DataAnnotations;
using UrlsHandlerSample.Services;

namespace UrlsHandlerSample.Pages
{
    public class IndexModel : PageModel
    {
        private readonly IPilotRegisterService _registerService;
        private readonly IHttpContextAccessor _httpContext;
        private readonly IPilotServiceConnector _pilotServiceConnector;

        public IndexModel(
            IPilotRegisterService registerService, 
            IHttpContextAccessor httpContext, 
            IPilotServiceConnector pilotServiceConnector)
        {
            _registerService = registerService;
            _httpContext = httpContext;
            _pilotServiceConnector = pilotServiceConnector;

            ConnectionFormModel = GetEmptyConnectionFormModel();
            RegisterFormModel = GetEmptyRegisterFormModel();
            RegisterSuccess = false;
            IsRegisterButtonEnabled = pilotServiceConnector.GetClient() != null;
        }
        
        [BindProperty]
        public bool RegisterSuccess { get; private set; }

        [BindProperty]
        public string? ConnectionError { get; private set; }

        [BindProperty]
        public string? Error { get; private set; }

        [BindProperty]
        public ConnectionFormModel ConnectionFormModel { get; set; }

        [BindProperty]
        public RegisterFormModel RegisterFormModel { get; set; }

        [BindProperty]
        public bool IsRegisterButtonEnabled { get;set; }

        [TempData]
        public string? ConnectionFormDataJson { get; set; }

        public IActionResult OnPostRegister() 
        {
            try
            {
                if (TempData.TryGetValue("ConnectionFormDataJson", out var json))
                {
                    var formData = (string)(json ?? "");
                    ConnectionFormModel 
                        =  JsonConvert.DeserializeObject<ConnectionFormModel>(formData)             
                        ?? GetEmptyConnectionFormModel();

                    TempData.Keep("ConnectionFormDataJson");
                }

                if (!IsRegisterButtonEnabled)
                    return Page();

                RegisterSuccess = false;
                Error = null;
                _registerService.RegisterHandler(RegisterFormModel.HandlerName, new Uri(RegisterFormModel.CallbackUrl));
                RegisterSuccess = true;
            }
            catch (Exception e)
            {
                Error = e.Message;
            }
            
            return Page();
        }

        public async Task<IActionResult> OnPostConnect()
        {
            try
            {
                Error = null;
                ConnectionError = null;
                RegisterFormModel = GetEmptyRegisterFormModel();

                await _pilotServiceConnector.Connect(
                    ConnectionFormModel.ServerAddress,
                    ConnectionFormModel.Database, 
                    ConnectionFormModel.Login, 
                    ConnectionFormModel.Password, 
                    ConnectionFormModel.License);

                IsRegisterButtonEnabled = _pilotServiceConnector.GetClient() != null;
                ConnectionFormDataJson = JsonConvert.SerializeObject(ConnectionFormModel);
                TempData.Keep("ConnectionFormDataJson");
            }
            catch (Exception e)
            {
                ConnectionError = e.Message;
            }

            return Page();
        }

        private ConnectionFormModel GetEmptyConnectionFormModel()
        {
            return new ConnectionFormModel
            {
                ServerAddress = string.Empty,
                Database = string.Empty,
                License = 100,
                Login = string.Empty,
                Password = string.Empty

            };
        }

        private RegisterFormModel GetEmptyRegisterFormModel()
        {
            var request = _httpContext?.HttpContext?.Request;
            return new RegisterFormModel()
            {
                CallbackUrl = request != null ? $"{request.Scheme}://{request.Host}/ObjectInfo" : string.Empty,
                HandlerName = string.Empty
            };
        }
    }

    public class ConnectionFormModel
    {
        [Required(ErrorMessage = "Enter Pilot-Server address")]
        public string ServerAddress { get; set; }
        [Required(ErrorMessage = "Enter database name")]
        public string Database { get; set; }
        [Required(ErrorMessage = "Enter license")]
        public int License { get; set; }
        [Required(ErrorMessage = "Enter login")]
        public string Login { get; set; }
        [Required(ErrorMessage = "Enter password")]
        public string Password { get; set; }
    }

    public class RegisterFormModel
    {
        [Required(ErrorMessage = "Enter callback url handler name")]
        public string HandlerName { get; set; }
        [Required(ErrorMessage = "Enter callback url")]
        public string CallbackUrl { get; set; }
    }
}
