/*
Copyright © 2024 ASCON-Design Systems LLC. All rights reserved.
This sample is licensed under the MIT License.
*/

export enum ErrorSeverity {
  NonError = "NonError",
  Warning = "Warning",
  Error = "Error",
}

export class CryptoProCSPError extends Error {
  private _errorCode: string;

  get errorCode(): string {
    return this._errorCode;
  }

  get severity(): ErrorSeverity {
    return this.classifyCSPErrorSeverity();
  }

  constructor(message: string) {
    super(message);
    this._errorCode = this.findErrorCode(message);
    Object.setPrototypeOf(this, CryptoProCSPError.prototype);
  }

  private findErrorCode(message: string): string {
    const regex = /0[xX][0-9a-fA-F]+/;
    const fullCodeMatches = message.match(regex);
    if (fullCodeMatches && fullCodeMatches.length > 0) {
      const fullCode = fullCodeMatches[0];
      return fullCode.replace(/[()]/g, "");
    }

    return "";
  }

  private classifyCSPErrorSeverity(): ErrorSeverity {
    if (this._errorCode in certCodeToSeverity) {
      return certCodeToSeverity[this._errorCode];
    }

    if (this._errorCode in miscCodeToSeverity) {
      return miscCodeToSeverity[this._errorCode];
    }

    return ErrorSeverity.Error;
  }
}

export const certCodeToSeverity: Record<string, ErrorSeverity> = {
  "0x800B0001": ErrorSeverity.Error,
  "0x800B0002": ErrorSeverity.Error,
  "0x800B0003": ErrorSeverity.Error,
  "0x800B0004": ErrorSeverity.Warning,
  "0x800B0005": ErrorSeverity.Error,
  "0x800B0006": ErrorSeverity.Error,
  "0x800B0007": ErrorSeverity.Warning,
  "0x800B0008": ErrorSeverity.Error,
  "0x800B0009": ErrorSeverity.Error,
  "0x800B000A": ErrorSeverity.Error,
  "0x800B000B": ErrorSeverity.Error,
  "0x800B0100": ErrorSeverity.Error,
  "0x800B0101": ErrorSeverity.Warning,
  "0x800B0102": ErrorSeverity.Warning,
  "0x800B0103": ErrorSeverity.Warning,
  "0x800B0104": ErrorSeverity.Warning,
  "0x800B0105": ErrorSeverity.Warning,
  "0x800B0106": ErrorSeverity.Warning,
  "0x800B0107": ErrorSeverity.Warning,
  "0x800B0108": ErrorSeverity.Warning,
  "0x800B0109": ErrorSeverity.Warning,
  "0x800B010A": ErrorSeverity.Warning,
  "0x800B010B": ErrorSeverity.Error,
  "0x800B010C": ErrorSeverity.Warning,
  "0x800B010D": ErrorSeverity.Warning,
  "0x800B010E": ErrorSeverity.Error,
  "0x800B010F": ErrorSeverity.Warning,
  "0x800B0110": ErrorSeverity.Warning,
  "0x800B0111": ErrorSeverity.Warning,
  "0x800B0112": ErrorSeverity.Warning,
  "0x800B0113": ErrorSeverity.Warning,
  "0x800B0114": ErrorSeverity.Warning,
};

export const miscCodeToSeverity: Record<string, ErrorSeverity> = {
  // Отмена в диалоге "Подтверждение доступа" ("Этот веб-сайт пытается выполнить операцию...")
  "0x000004C7": ErrorSeverity.NonError,
  // Отмена в диалоге "Аутентификация - КриптоПро CSP", по https://bugs.pilot.ascon.net/issue/3-2250
  "0x8010006E": ErrorSeverity.NonError,
  // Закончилась лицензия на КриптоПро CSP - отказ активировать новую
  "0x8007065B": ErrorSeverity.NonError,
};

