import { Asn1Parser, AsnObject } from "./asn.parser";
// @ts-ignore
import { Base64 } from '@lapo/asn1js/base64.js';
import { CadesType } from "@pilotdev/pilot-web-sdk";

export interface ICertificateInfo {
  commonName: string | null;
  organizationName: string | null;
  localityName: string | null;
  countryName: string | null;
  emailAddress: string | null;
  notBeforeDate: string | null;
  notAfterDate: string | null;
  serialNumber: string;
  stateOrProvinceName?: string | null;
  streetAddress?: string | null;
  surname?: string | null;
  givenName?: string | null;
  title?: string | null;
}

export interface IParsedSignature {
  subject: ICertificateInfo;
  signatureAlgorithm: string;
  cadesType: CadesType,
  signingDate: string | null,
  timeStampDate?: string | null,
}

export class SignatureParser {
  
  static parse(sign: string): IParsedSignature {
    const asnObj = Asn1Parser.initAsn(Base64.unarmor(sign));

    var certChain = this.collectCertificatesChain(asnObj);
    const certificateSubject = certChain![certChain!.length - 1];
    return {
      subject: certificateSubject,
      timeStampDate: this.getTimeStampDate(asnObj),
      signingDate: this.getSigningDate(asnObj),
      signatureAlgorithm: this.getAlgorithm(asnObj),
      cadesType: this.getCadesType(asnObj)
    };
  }
  static getCadesType(asnObj: AsnObject): CadesType {
    const signerInfos = Asn1Parser.searchInTree(asnObj, 'signerInfos');
    const contentTypeBlock = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.3', signerInfos as AsnObject, 'name');
    const messageDigestBlock = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.4', signerInfos as AsnObject, 'name');
    const signingCertificateV2 = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.16.2.47', signerInfos as AsnObject, 'name');

    const timeStampToken = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.16.2.14', signerInfos as AsnObject, 'name');

    const revocationRefs = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.16.2.22', signerInfos as AsnObject, 'name');
    const certificateRefs = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.16.2.21', signerInfos as AsnObject, 'name');

    const revocationValues = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.16.2.24', signerInfos as AsnObject, 'name');
    const certValues = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.16.2.23', signerInfos as AsnObject, 'name');

    const escTimeStamp = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.16.2.25', signerInfos as AsnObject, 'name');
    const escCertTimeStamp = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.16.2.26', signerInfos as AsnObject, 'name');

    const archiveTimeStamp = Asn1Parser.searchInBlock(signerInfos as AsnObject, 'AttributeType', '1.2.840.113549.1.9.16.2.27', signerInfos as AsnObject, 'name');


    const isCadesBes = contentTypeBlock && messageDigestBlock && signingCertificateV2;
    const isCadesT = isCadesBes && timeStampToken;
    const isCadesC = isCadesT && revocationRefs && certificateRefs;
    const isCadesXLong = isCadesC && revocationValues && certValues;
    const isCadesXLongType1 = isCadesXLong && escTimeStamp;
    const isCadesXLongType2 = isCadesXLong && escCertTimeStamp;
    const isCadesA = (isCadesXLongType1 || isCadesXLongType2) && archiveTimeStamp;

    if (isCadesA)
      return CadesType.CadesA;

    if (isCadesXLongType2)
      return CadesType.CadesXLongType2;

    if (isCadesXLongType1)
      return CadesType.CadesXLongType1;

    if (isCadesC)
      return CadesType.CadesC;

    if (isCadesT)
      return CadesType.CadesT;

    if (isCadesBes)
      return CadesType.CadesBes;

    return CadesType.NotCades;
  }

  static getAlgorithm(node: AsnObject): string {
    const signerInfos = Asn1Parser.searchInTree(node, 'signerInfos');
    const algorithm = Asn1Parser.searchInTree(signerInfos as AsnObject, `signatureAlgorithm`);
    const name = Asn1Parser.searchInTree(algorithm as AsnObject, "algorithm");
    const algParts = name?.content()?.split('\n');
    return algParts[0];
  }


  private static getTimeStampDate(node: AsnObject): string | null {
    const timeStampToken = Asn1Parser.searchInBlock(node, 'AttributeType', '1.2.840.113549.1.9.16.2.14', node, 'name');

    if (timeStampToken) {
      const generalizedTagNumber = 0x18;
      const timeStampSet = Asn1Parser.searchInTree(timeStampToken, 'values');
      const timeStampValue = Asn1Parser.searchByTag(timeStampSet as AsnObject, generalizedTagNumber, false)?.content() || null;

      return timeStampValue;
    }

    return null;
  }

  static arrayBufferToDecodedString(buffer: ArrayBuffer): string {
    const arrayBuffer = new Uint8Array(buffer).buffer;
    const decoder = new TextDecoder("utf-8");
    return decoder.decode(arrayBuffer);
  }

  static getSigningDate(node: AsnObject): string | null {
    const signedAttributes = Asn1Parser.searchInTree(node, 'SignedAttributes', 'name');

    if (signedAttributes) {
      const signingBlock = Asn1Parser.searchInBlock(signedAttributes, 'AttributeType', '1.2.840.113549.1.9.5', signedAttributes, 'name');
      const signingSet = Asn1Parser.searchInTree(signingBlock as AsnObject, 'values');
      const result = Asn1Parser.searchInTree(signingSet as AsnObject, 'AttributeValue', 'name')?.content() || null;

      return result;
    }

    return null;
  }

  private static collectCertificatesChain(node: AsnObject): ICertificateInfo[] | null {
    let certificateList: ICertificateInfo[] = [];

    const certificateSet = Asn1Parser.searchInTree(node, 'CertificateSet', 'name');
    if (certificateSet && certificateSet.sub && certificateSet.sub.length) {
      const certificateChoices = certificateSet.sub.filter(item => item && item.def.name === 'CertificateChoices');

      certificateChoices.forEach(item => {
        const certSubject = this.getCertificateInfo(item, 'subject');
        if (certSubject)
          certificateList.push(certSubject);
      });

      if (certificateList.length)
        certificateList = certificateList.filter((item, index) => {
          const { notBeforeDate, notAfterDate, serialNumber, ...itemClone } = item;
          return index === certificateList.findIndex(cert => {
            const { notBeforeDate, notAfterDate, serialNumber, ...certClone } = cert;
            return JSON.stringify(itemClone) === JSON.stringify(certClone);
          });
        });
    }

    certificateList.forEach(item => {

      if (item.serialNumber)
        item.serialNumber = this.convertSerialNumber(item.serialNumber);
    });

    return certificateList.length ? certificateList : null;
  }

  private static convertSerialNumber(decimalSerialNumber: string): string {
    return decimalSerialNumber && decimalSerialNumber.trim() ? BigInt(decimalSerialNumber).toString(16).toUpperCase() : '';
  }

  private static getCertificateInfo(certificateBlock: AsnObject, blockName: string = 'issuer'): ICertificateInfo | null {
      const dataBlock = Asn1Parser.searchInTree(certificateBlock, blockName);
  
      if (dataBlock && certificateBlock) {
        const issuerTypeValueBlock = Asn1Parser.searchInBlock(dataBlock, 'AttributeType', '2.5.4.3', certificateBlock, 'name');
        const organizationNameBlock = Asn1Parser.searchInBlock(dataBlock, 'AttributeType', '2.5.4.10', certificateBlock, 'name');
        const localityNameBlock = Asn1Parser.searchInBlock(dataBlock, 'AttributeType', '2.5.4.7', certificateBlock, 'name');
        const stateOrProvinceNameBlock = Asn1Parser.searchInBlock(dataBlock, 'AttributeType', '2.5.4.8', certificateBlock, 'name');
        const streetAddressNameBlock = Asn1Parser.searchInBlock(dataBlock, 'AttributeType', '2.5.4.9', certificateBlock, 'name');
        const countryNameBlock = Asn1Parser.searchInBlock(dataBlock, 'AttributeType', '2.5.4.6', certificateBlock, 'name');
        const emailAddressBlock  = Asn1Parser.searchInBlock(dataBlock, 'AttributeType', '1.2.840.113549.1.9.1', certificateBlock, 'name');
        const surnameBlock = Asn1Parser.searchInBlock(dataBlock, 'AttributeType', '2.5.4.4', certificateBlock, 'name');
        const givenNameBlock = Asn1Parser.searchInBlock(dataBlock, 'AttributeType', '2.5.4.42', certificateBlock, 'name');
        const titleNameBlock = Asn1Parser.searchInBlock(dataBlock, 'AttributeType', '2.5.4.12', certificateBlock, 'name');
  
        const commonName = Asn1Parser.searchInTree(issuerTypeValueBlock as AsnObject, 'value')?.content() || null;
        const organizationName = Asn1Parser.searchInTree(organizationNameBlock as AsnObject, 'value')?.content() || null;
        const localityName = Asn1Parser.searchInTree(localityNameBlock as AsnObject, 'value')?.content() || null;
        const stateOrProvinceName = Asn1Parser.searchInTree(stateOrProvinceNameBlock as AsnObject, 'value')?.content() || null;
        const streetAddress = Asn1Parser.searchInTree(streetAddressNameBlock as AsnObject, 'value')?.content() || null;
        const countryName = Asn1Parser.searchInTree(countryNameBlock as AsnObject, 'value')?.content() || null;
        const emailAddress = Asn1Parser.searchInTree(emailAddressBlock as AsnObject, 'value')?.content() || null;
        const surname = Asn1Parser.searchInTree(surnameBlock as AsnObject, 'value')?.content() || null;
        const givenName = Asn1Parser.searchInTree(givenNameBlock as AsnObject, 'value')?.content() || null;
        const title = Asn1Parser.searchInTree(titleNameBlock as AsnObject, 'value')?.content() || null;
  
        const serialNumber = Asn1Parser.searchInTree(certificateBlock, 'serialNumber')?.content()?.split('\n')[1] || null;
  
        return {
          commonName,
          organizationName,
          localityName,
          stateOrProvinceName,
          streetAddress,
          countryName,
          emailAddress,
          serialNumber,
          surname,
          givenName,
          title,
          ...this.getValidity(certificateBlock)
        };
      }
  
      return null;
    } 

    private static getValidity(certificateChoicesBlock: AsnObject): { notBeforeDate: string | null, notAfterDate: string | null }  {
      const validity = Asn1Parser.searchInTree(certificateChoicesBlock, 'validity');
  
      if (validity && validity.sub && validity.sub[0]) {
        const notBeforeDate = Asn1Parser.searchInTree(validity.sub[0], 'Time', 'name')?.content() || null;
        const notAfterDate = Asn1Parser.searchInTree(validity.sub[1], 'Time', 'name')?.content() || null;
  
        return {
          notBeforeDate,
          notAfterDate
        };
      }
  
      return {
                notBeforeDate: null,
                notAfterDate: null 
            };
    }
  
}