/*
Copyright © 2024 ASCON-Design Systems LLC. All rights reserved.
This sample is licensed under the MIT License.
*/
import { Guid, ICertificate, IDataObject, IModifierProvider, CadesType, IFile } from "@pilotdev/pilot-web-sdk";
import { Observable } from "rxjs";
import { convertToArrayBuffer, getCNPart } from "./utils/utils";
import { IAdvancementSettingsProvider } from "./advancement/advancementSettingsProvider";

export class FileSignatureUpdater {

  constructor(
    private readonly _modifierProvider: IModifierProvider,
    private readonly _enhancementSettings: IAdvancementSettingsProvider) {
  }

  setSignToObjectFile(documentId: string, actualFile: IFile, signatureBase64: string, publicKeyOid: string, signatureRequestIds: string[], certificate: ICertificate): Observable<IDataObject[]> {
    const modifier = this._modifierProvider.newModifier();
    const builder = modifier.edit(documentId);

    signatureRequestIds.forEach(requestId => {
      const requestToSign = actualFile.signatureRequests.find(req => req.id === requestId);
      if (!requestToSign) 
        return;

      const fileId = Guid.newGuid();
        const creationDate = new Date(); // UTC;
        const fileBuffer = convertToArrayBuffer(signatureBase64);
        const file: File = new File([fileBuffer], `${actualFile.name}.${requestId}.sig`, {
          type: 'application/octet-stream' ,
          lastModified: creationDate.getUTCDate()
        });
        const certificateCN = getCNPart(certificate.issuer);
        const setting = this._enhancementSettings.getSettings()
          .find(setting => this.removeQuotes(setting) === this.removeQuotes(certificateCN));
        
        const advancementRequired = !!setting;

        builder
          .addFile(fileId, file, creationDate, creationDate, creationDate)
          .setSignatures(actualFile.body.id)
            .edit(requestToSign)
            .withSign(fileId)
            .withPublicKeyOid(publicKeyOid)
            .withLastSignCadesType(CadesType.CadesBes)
            .withIsAdvancementRequired(advancementRequired);
    });

    return modifier.apply();
  }

  private removeQuotes(str: string): string {
    return str.replaceAll('"', '');
  }
}