import { AfterViewInit, Component, Input, ViewChild } from '@angular/core';
import { IFile, IFileSnapshot } from "@pilotdev/pilot-web-sdk";
import {
  MatCell,
  MatCellDef,
  MatColumnDef,
  MatHeaderCell, MatHeaderCellDef,
  MatHeaderRow,
  MatHeaderRowDef, MatRow, MatRowDef, MatTable, MatTableDataSource
} from "@angular/material/table";
import { MatTooltip } from "@angular/material/tooltip";
import { DatePipe } from "@angular/common";
import { MatButton, MatIconButton } from "@angular/material/button";
import { MatIconModule } from "@angular/material/icon";
import { SelectionModel } from "@angular/cdk/collections";
import { MatToolbar, MatToolbarModule } from "@angular/material/toolbar";
import { SignaturesTableComponent } from "../signatures-table/signatures-table.component";
import { MatPaginator } from "@angular/material/paginator";
import { DownloadFilesService } from "../services/download-files.service";

@Component({
  selector: 'app-files',
  standalone: true,
  imports: [
    MatCell,
    MatCellDef,
    MatColumnDef,
    MatHeaderCell,
    MatHeaderRow,
    MatHeaderRowDef,
    MatRow,
    MatRowDef,
    MatTable,
    MatHeaderCellDef,
    MatTooltip,
    DatePipe,
    MatButton,
    MatIconModule,
    MatIconButton,
    MatToolbar,
    MatToolbarModule,
    SignaturesTableComponent,
    MatPaginator
  ],
  providers: [ DownloadFilesService],
  templateUrl: './files.component.html',
  styleUrl: './files.component.scss'
})
export class FilesComponent implements AfterViewInit {
  private _fileSelectionModel = new SelectionModel<IFile>(true, []);
  private _snapshot!: IFileSnapshot;
  private _lastSelectedIndex: number | null = null;

  public filesDatasource!: MatTableDataSource<IFile>;
  public displayedColumns: string[] = ["id", "name", "created", "accessed", "size", "md5", "download"];

  public get selectedFile(): IFile {
    return this._fileSelectionModel.selected[0];
  }

  public get snapshot(): IFileSnapshot {
    return this._snapshot;
  }

  @Input() set snapshot(snapshot: IFileSnapshot) {
    this._fileSelectionModel.clear();
    this._snapshot = snapshot;
    this.filesDatasource = new MatTableDataSource<IFile>(this._snapshot?.files);
  }

  @Input() objectId!: string;

  @ViewChild(MatPaginator) paginator!: MatPaginator;

  constructor(
    private readonly _downloadFilesService: DownloadFilesService,
  ) {}

  ngAfterViewInit() {
    this.filesDatasource.paginator = this.paginator;
  }

  public onFileItemSelect(event: MouseEvent, row: IFile, rowIndex: number): void {
    if ((event.shiftKey || event.ctrlKey) && this._lastSelectedIndex !== null) {
      // множественный выбор через Shift
      const start = Math.min(this._lastSelectedIndex, rowIndex);
      const end = Math.max(this._lastSelectedIndex, rowIndex);
      const rowsToSelect = this.filesDatasource.data.slice(start, end + 1);

      rowsToSelect.forEach(r => this._fileSelectionModel.select(r));
    } else {
      // обычный одиночный выбор
      this._fileSelectionModel.clear();
      this._fileSelectionModel.select(row);
      this._lastSelectedIndex = rowIndex;
    }
  }

  public isSelected(row: IFile): boolean {
    return this._fileSelectionModel.isSelected(row);
  }

  public download(file: IFile): void {
    this._downloadFilesService.download(file);
  }

  public downloadAll(): void {
    this._downloadFilesService.downloadFileArchive([this.objectId], this._snapshot.created);
  }
}
