﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using PdfSharp.Drawing;
using PdfSharp.Pdf.Annotations;
using PdfSharp.Pdf;
using PdfSharp.Fonts;
using PdfStamper.Resources;
using PdfStamper.Certificates;
using PdfSharp.Drawing.Layout;
using Pilot.WebServer.SDK;
using PdfStamper.Settings;
using PdfSharp.Pdf.IO;

namespace PdfStamper;

public enum StampType
{
    Signature,
    IntoProduction
}

class StampCreator : IDisposable
{
    private readonly PdfDocument _document;
    private readonly PdfStamperMode _stamperMode;
    private readonly SettingsModel _settingsModel;
    private readonly Stream _documentStream;

    public StampCreator(Stream originalStream, PdfStamperMode stamperMode, SettingsModel settingsModel)
    {
        GlobalFontSettings.FontResolver = new FontResolver();
        _stamperMode = stamperMode;
        _documentStream = originalStream;
        _document = PdfReader.Open(_documentStream, PdfDocumentOpenMode.Modify);
        _documentStream.Position = 0;
        _settingsModel = settingsModel;
    }

    public int GetPagesCount()
    {
        return _document.PageCount;
    }

    public XSize GetPageSize(int pageNumber)
    {
        if (_document.Pages.Count < pageNumber - 1)
            throw new ArgumentOutOfRangeException(nameof(pageNumber));

        var height = _document.Pages[pageNumber].Height;
        var width = _document.Pages[pageNumber].Width;

        return new XSize(width.Point, height.Point);
    }

    public void SaveStamps()
    {
        _document.Save(_documentStream, false);
    }

    public void Dispose()
    {
        _document.Dispose();
    }

    public void AddStamp(int pageNumber, CertificateInfo certInfo, StampPosition stampPosition, XSize stampSize, StampType stampType)
    {
        var page = _document.Pages[pageNumber];

        using var pageGfx = XGraphics.FromPdfPage(page);
        var startPosition = new XPoint(stampPosition.X, stampPosition.Y);
        var xRect = new XRect(startPosition, stampSize);
        var stampRect = pageGfx.Transformer.WorldToDefaultPage(xRect);

        XForm? form = null;
        XGraphics drawGfx;

        if (_stamperMode == PdfStamperMode.Preview)
        {
            form = new XForm(_document, stampSize);
            drawGfx = XGraphics.FromForm(form);
            startPosition = new XPoint(0, 0);
        }
        else
        {
            drawGfx = pageGfx;
        }

        using (drawGfx)
        {
            var font = new XFont(Constants.FONT_NAME, 7, XFontStyleEx.Regular);
            DrawRoundedRectangle(drawGfx, startPosition, stampSize);

            if (stampType == StampType.Signature)
                DrawSignatureStamp(certInfo, stampSize, drawGfx, font, startPosition);

            if (stampType == StampType.IntoProduction)
                DrawInProductionStamp(certInfo, stampSize, drawGfx, font, startPosition);
        }

        if (_stamperMode == PdfStamperMode.Preview)
            AddStampToPage(certInfo, page, stampRect, form);
    }

    private void AddStampToPage(CertificateInfo certInfo, PdfPage page, XRect stampRect, XForm? form)
    {
        var stamp = new PdfRubberStampAnnotation { Rectangle = new PdfRectangle(stampRect) };
        SetAppearance(form!, stamp);
        SetId(stamp, certInfo.Id!);
        page.Annotations.Add(stamp);
    }

    private void DrawSignatureStamp(CertificateInfo certInfo, XSize stampSize, XGraphics drawGfx, XFont font, XPoint currentPosition)
    {
        var headerHeight = 28;
        DrawHeader(drawGfx, font, stampSize, headerHeight, currentPosition);
        
        // обновляем текущую позицию по y. Следующий элемент рисуем ниже на высоту заголовка
        currentPosition.Y += headerHeight;
        DrawCertificateInfo(drawGfx, font, certInfo, currentPosition, stampSize);
        currentPosition.Y -= headerHeight;
        
        if (_settingsModel.Image != null)
        {
            using var imgStream = new MemoryStream();
            imgStream.Write(_settingsModel.Image, 0, _settingsModel.Image.Length);
            using var img = XImage.FromStream(imgStream);
            var imgWidth = 32;
            var imgHeight = 32;
            drawGfx.DrawImage(img, currentPosition.X + 20, currentPosition.Y + 4, imgWidth, imgHeight);
        }
    }

    private void DrawInProductionStamp(CertificateInfo certInfo, XSize stampSize, XGraphics drawGfx, XFont font, XPoint currentPos)
    {
        var boldFont = new XFont(Constants.FONT_NAME, 12, XFontStyleEx.Bold);
        var padding = 5;
        currentPos.Y += padding * 4;
        drawGfx.DrawString(Constants.INTO_PRODUCTION, boldFont, XBrushes.Black, new XPoint(currentPos.X + padding * 7, currentPos.Y));

        currentPos.Y += padding;
        DrawCertificateInfo(drawGfx, font, certInfo, currentPos, stampSize);
    }

    private void DrawCertificateInfo(XGraphics gfx, XFont font, CertificateInfo certInfo, XPoint currentPosition, XSize stampSize)
    {
        // заголовок на черном фоне
        var borderPadding = 1;
        var startPosition = new XPoint(currentPosition.X + borderPadding, currentPosition.Y + 2);

        // Рисуем рамку c заголовком ЭЦП под картинкой
        var headerRect = new XRect(startPosition.X, startPosition.Y, stampSize.Width - (borderPadding * 2), 10);
        gfx.DrawRectangle(XPens.Black, XBrushes.Black, headerRect);
        gfx.DrawString("СВЕДЕНИЯ О СЕРТИФИКАТЕ ЭП", font, XBrushes.White, headerRect, XStringFormats.Center);

        // Текст после заголовка 
        var textMarginTop = 8;
        startPosition.Y += headerRect.Height + textMarginTop;
        startPosition.X += 6;
        gfx.DrawString($"Владелец: {certInfo.Subject}", font, XBrushes.Black, new XPoint(startPosition.X, startPosition.Y));
        startPosition.Y += font.Height;
        gfx.DrawString($"Сертификат: {certInfo.Thumbprint}", font, XBrushes.Black, new XPoint(startPosition.X, startPosition.Y));
        startPosition.Y += font.Height;
        gfx.DrawString($"Действителен: с {certInfo.ValidFrom} до {certInfo.ValidTo}", font, XBrushes.Black, new XPoint(startPosition.X, startPosition.Y));
    }

    private void DrawRoundedRectangle(XGraphics gfx, XPoint startPosition,  XSize stampSize)
    {
        var mainRect = new XRect(startPosition.X + 1, startPosition.Y + 1, stampSize.Width - 2, stampSize.Height - 2);
        var ellipseSize = new XSize(5, 5);
        gfx.DrawRoundedRectangle(XPens.Black, XBrushes.Transparent, mainRect, ellipseSize);
    }

    private void DrawHeader(XGraphics gfx, XFont font, XSize stampSize, double headerHeight, XPoint currentPosition)
    {
        var headerMarginLeft = 62;
        var headerMarginTop = 3;
        var headerStartPositionX = currentPosition.X + headerMarginLeft;
        var headerStartPositionY = currentPosition.Y + headerMarginTop;

        var tf = new XTextFormatter(gfx) { Alignment = XParagraphAlignment.Left };
        var rect = new XRect(headerStartPositionX, headerStartPositionY, stampSize.Width - 62, headerHeight);
        
        // Рисуем текст рядом с логотипом
        tf.DrawString(_settingsModel.StampHeader!, font, XBrushes.Black, rect);
    }

    private void SetId(PdfDictionary stamp, string stampId)
    {
        if (stamp == null)
            return;

        // Удалим стандартное имя
        stamp.Elements.SetString("/Name", "#" + Guid.NewGuid());
        // Присваиваем уникальный идентификатор через ключ /NM
        stamp.Elements.SetString("/NM", stampId);
    }

    private void SetAppearance(XForm form, PdfDictionary stamp)
    {
        if (form == null)
            return;

        // Устанавливаем appearance‑stream аннотации
        var appearanceStream = new PdfDictionary(_document);
        appearanceStream.Elements.Add("/N", form.GetPdfFormXObject());
        stamp.Elements.Add("/AP", appearanceStream);
    }
}
