using Pilot.WebServer.SDK;
using Pilot.WebServer.SDK.Data;
using System.ComponentModel.Composition;

namespace ProjectCloneWizard
{
  [Export(typeof(ICommandHost))]
  public class ProjectCloneWizard : ICommandHost
  {
    private const string DIGITAL_SIGNATURE_FILE_NAME = "PilotDigitalSignature";
    private const string REMARKS_FOLDER_TYPE_NAME = "doc_remarksFolder";
    private const string BIM_REMARKS_FOLDER_TYPE_NAME = "bim_modelRemarksFolder";
    private IModifierProvider _modifierProvider;
    private IObjectsRepository _repository;
    private IFileLoader _fileLoader;

    [ImportingConstructor]
    public ProjectCloneWizard(IModifierProvider modifierProvider, IObjectsRepository repository, IFileLoader fileLoader)
    {
      _modifierProvider = modifierProvider;
      _repository = repository;
      _fileLoader = fileLoader;
    }
    [Command]
    public void CopyStructure(Guid structureRoot, Guid destinaion)
    {
      var modifier = _modifierProvider.NewModifier();
      CopyObject(structureRoot, modifier, destinaion);
      modifier.Apply();
    }

    public void Invoked(string methodName)
    {
    }

    private void CopyObject(Guid originId, IModifier modifier, Guid parentId)
    {
      var origin = _repository.GetObjects(new Guid[] { originId }).First();

      if (!CanCopy(origin))
        return;

      var copyId = Guid.NewGuid();
      var builder = modifier.Create(copyId, parentId, SelectType(origin.Type).Id);

      if (origin.SecretInfo.SecretParentId != Guid.Empty)
        builder.SetIsSecret(true);

      SetAccess(origin.Access, builder);

      foreach (var attribute in origin.Attributes)
        SetAttribute(attribute.Key, attribute.Value, builder);

      foreach (var file in origin.ActualFileSnapshot.Files)
        SetFile(file, builder);

      foreach (var child in origin.Children)
        CopyObject(child.ObjectId, modifier, copyId);
    }

    private bool CanCopy(IObject origin)
    {
      if (origin.Type.Name == REMARKS_FOLDER_TYPE_NAME ||
          origin.Type.Name == BIM_REMARKS_FOLDER_TYPE_NAME)
        return false;

      return true;
    }

    private IType SelectType(IType type)
    {
      if (type.IsDeleted)
        return _repository.GetType(type.Name);
      return type;
    }

    private void SetFile(IFile file, IObjectBuilder builder)
    {
      if (file.Name == DIGITAL_SIGNATURE_FILE_NAME)
        return;

      var body = _fileLoader.Download(file.Id, file.Size);
      builder.AddFile(file.Name, new MemoryStream(body), file.Created, file.Accessed, file.Modified);
    }

    private void SetAttribute(string key, object value, IObjectBuilder builder)
    {
      switch (value)
      {
        case string val:
          builder.SetAttribute(key, val);
          break;
        case int val:
          builder.SetAttribute(key, val);
          break;
        case double val:
          builder.SetAttribute(key, val);
          break;
        case DateTime val:
          builder.SetAttribute(key, val);
          break;
        case long val:
          builder.SetAttribute(key, val);
          break;
        case Guid val:
          builder.SetAttribute(key, val);
          break;
        case int[] val:
          builder.SetAttribute(key, val);
          break;
        case string[] val:
          builder.SetAttribute(key, val);
          break;
        default:
          break;
      }
    }

    private void SetAccess(IEnumerable<IAccessRecord> accessRecords, IObjectBuilder builder)
    {
      foreach (var record in accessRecords)
      {
        builder.AddAccessRecord(record.OrgUnitId, record.Access.AccessLevel, record.Access.ValidThrough, record.Access.Inheritance, record.Access.Type, record.Access.TypeIds);
      }
    }

  }


}
