/**
 * Copyright (c) 2024 ASCON-Design Systems LLC. All rights reserved.
 * This sample is licensed under the MIT License.
 */
import {
  AfterViewInit,
  Component,
  ElementRef,
  Inject,
  OnDestroy,
  Renderer2,
  ViewChild,
  ViewEncapsulation,
} from '@angular/core';

import {
  CdkDragDrop,
  DragDropModule,
  moveItemInArray,
} from '@angular/cdk/drag-drop';
import { FormsModule } from '@angular/forms';
import { MatFormFieldModule } from '@angular/material/form-field';
import { MatIconModule } from '@angular/material/icon';
import { MatInputModule } from '@angular/material/input';
import { MatSort } from '@angular/material/sort';
import {
  MatTable,
  MatTableDataSource,
  MatTableModule,
} from '@angular/material/table';

import { IThemeService, Theme } from '@pilotdev/pilot-web-sdk';

export interface PeriodicElement {
  name: string;
  position: number;
  weight: number;
  symbol: string;
  quantity: number;
}

const ELEMENT_DATA: PeriodicElement[] = [
  { position: 1, name: 'Hydrogen', weight: 1.0079, symbol: 'H', quantity: 0 },
  { position: 2, name: 'Helium', weight: 4.0026, symbol: 'He', quantity: 0 },
  { position: 3, name: 'Lithium', weight: 6.941, symbol: 'Li', quantity: 0 },
  { position: 4, name: 'Beryllium', weight: 9.0122, symbol: 'Be', quantity: 0 },
  { position: 5, name: 'Boron', weight: 10.811, symbol: 'B', quantity: 0 },
  { position: 6, name: 'Carbon', weight: 12.0107, symbol: 'C', quantity: 0 },
  { position: 7, name: 'Nitrogen', weight: 14.0067, symbol: 'N', quantity: 0 },
  { position: 8, name: 'Oxygen', weight: 15.9994, symbol: 'O', quantity: 0 },
  { position: 9, name: 'Fluorine', weight: 18.9984, symbol: 'F', quantity: 0 },
  { position: 10, name: 'Neon', weight: 20.1797, symbol: 'Ne', quantity: 0 },
];

@Component({
  selector: 'app-table-demo',
  standalone: true,
  imports: [
    MatTableModule,
    MatInputModule,
    DragDropModule,
    MatFormFieldModule,
    MatIconModule,
    FormsModule,
  ],
  templateUrl: './table-demo.component.html',
  styleUrl: './table-demo.component.scss',
  encapsulation: ViewEncapsulation.ShadowDom,
})
export class TableDemoComponent implements AfterViewInit, OnDestroy {
  @ViewChild('table', { static: true }) table!: MatTable<PeriodicElement>;
  @ViewChild(MatSort) sort!: MatSort;
  @ViewChild('wrapperElement') wrapperDiv!: ElementRef<HTMLDivElement>;

  displayedColumns: string[] = [
    'position',
    'name',
    'weight',
    'symbol',
    'quantity',
  ];
  dataSource: MatTableDataSource<PeriodicElement>;
  dragDisabled = true;

  constructor(
    @Inject('IThemeService') private _themeService: IThemeService,
    private _renderer: Renderer2
  ) {
    this.dataSource = new MatTableDataSource([...ELEMENT_DATA]);
  }

  ngAfterViewInit(): void {
    this._themeService.change.subscribe((theme) => {
      if (theme.theme === Theme.Dark) {
        this._renderer.addClass(this.wrapperDiv.nativeElement, 'theme-dark');
      } else {
        this._renderer.removeClass(this.wrapperDiv.nativeElement, 'theme-dark');
      }
    });

    this.dataSource.sort = this.sort;
  }

  ngOnDestroy(): void {
    this.dataSource.data = [];
    this.dataSource.sort = null;
  }

  applyFilter(event: Event) {
    const filterValue = (event.target as HTMLInputElement).value;
    this.dataSource.filter = filterValue.trim().toLowerCase();
  }

  getTotalQty() {
    return this.dataSource.data
      .map((t) => t.quantity)
      .reduce((acc, value) => acc + value, 0);
  }

  drop(event: CdkDragDrop<string>) {
    this.dragDisabled = true;

    const previousIndex = this.dataSource.data.findIndex(
      (d) => d === event.item.data
    );

    moveItemInArray(this.dataSource.data, previousIndex, event.currentIndex);
    this.dataSource.data = [...this.dataSource.data];
    this.table.renderRows();
  }
}
